/**
 * @brief 一个多API实例的报撤单演示
 *
 * 注意，本示例代码演示的是一种错误用法：
 * 1. API-1登录账号AccountID-1；
 * 2. API-2登录账号AccountID-2；
 * 3. 通过API-1查询某个合约InstrumentID的对象指针；
 * 4. 使用该合约对象构造报单对象InputOrder；
 * 5. 通过API-2发送构造好的报单对象；
 *
 * 虽然报单是通过API-2（AccountID-2）发送的，但是合约对象属于API-1（AccountID-1）；
 * 因此，报单依然属于AccountID-1的报单，且通过AccountID-1的回调接口接收回报。
 *
 * 演示功能：
 * 1. 通过两份配置文件创建两个API实例；
 * 2. 两个API分别连接和登录柜台；
 * 3. 等数据加载完毕后，从API-1查询合约对象，并使用API-2发送报单，验证报单回报从API-1返回；
 * 4. 等待3秒后，撤销未成交报单；
 * 5. 等待3秒后，登出柜台；
 */

#include <map>
#include <XTFErrorCode.h>
#include "ExampleTrader.h"

class Example_07_Trader : public ExampleTrader {
public:
    Example_07_Trader() = default;
    ~Example_07_Trader() = default;

    void onStart(int errorCode, bool isFirstTime) override {
        ExampleTrader::onStart(errorCode, isFirstTime);
        printf("%s: api start ok.\n", mTraderName.c_str());
    }

    void onStop(int reason) override {
        ExampleTrader::onStop(reason);
        printf("%s: api stopped, reason: %d.\n", mTraderName.c_str(), reason);
    }

    void onServerReboot() override {
        printf("%s: trade system reboot, clear local data.\n", mTraderName.c_str());
        mOrders.clear();
    }

    void onLogin(int errorCode, int exchangeCount) override {
        ExampleTrader::onLogin(errorCode, exchangeCount);
        if (errorCode != 0)
            printf("%s: login failed, error code: %d.\n", mTraderName.c_str(), errorCode);
        else
            printf("%s: login success, exchange count=%d.\n", mTraderName.c_str(), exchangeCount);
    }

    void onLogout(int errorCode) override {
        ExampleTrader::onLogout(errorCode);
        printf("%s: logout success.\n", mTraderName.c_str());
    }

    void onReadyForTrading(const XTFAccount *account) override {
        ExampleTrader::onReadyForTrading(account);
        printf("%s: ready for trading.\n", mTraderName.c_str());
        mOrderLocalId = account->lastLocalOrderID;
    }

    void onLoadFinished(const XTFAccount *account) override {
        ExampleTrader::onLoadFinished(account);
        printf("%s: load data finished.\n", mTraderName.c_str());
    }

    void onOrder(int errorCode, const XTFOrder *order) override {
        printf("%s: recv order report: "
               "action=%d, local-id=%d, sys-id=%d, "
               "status=%s, error-code=%d.\n",
               mTraderName.c_str(),
               order->actionType, order->localOrderID, order->sysOrderID,
               getOrderStatus(order->orderStatus), errorCode);
        if (errorCode == 0) {
            switch (order->orderStatus) {
                case XTF_OS_Queuing:
                    mOrders[order->localOrderID] = order->sysOrderID;
                    break;
                case XTF_OS_AllTraded:
                case XTF_OS_Rejected:
                    mOrders.erase(order->localOrderID);
                    break;
                default:
                    break;
            }
        }
    }

    void onCancelOrder(int errorCode, const XTFOrder *cancelOrder) override {
        printf("%s: recv cancel order report: "
               "local-id=%d, sys-id=%d, status=%s, error-code=%d.\n",
               mTraderName.c_str(),
               cancelOrder->localOrderID, cancelOrder->sysOrderID,
               getOrderStatus(cancelOrder->orderStatus), errorCode);
        if (errorCode == 0 || errorCode == 1198) {
            mOrders.erase(cancelOrder->localOrderID); // 如果报单被撤后，从本地列表中删除该报单信息
        }
    }

    void onTrade(const XTFTrade *trade) override {
        printf("%s: recv trade report: "
               "trade-id=%ld, price=%.4f, "
               "volume=%d/%d, sys-order-id=%d\n",
               mTraderName.c_str(),
               trade->tradeID, trade->tradePrice,
               trade->order->totalTradedVolume, trade->order->orderVolume,
               trade->order->sysOrderID);
    }

    void start() {
        if (mApi) {
            printf("%s: error: trader has been started.\n", mTraderName.c_str());
            return;
        }

        mOrderLocalId = 0;
        mApi = makeXTFApi(mConfigPath.c_str());
        if (mApi == nullptr) {
            printf("%s: error: create xtf api failed, please check config: %s.\n",
                   mTraderName.c_str(), mConfigPath.c_str());
            exit(0);
        }

        printf("%s: api starting..., config: %s.\n", mTraderName.c_str(), mConfigPath.c_str());
        mApi->start(this);
    }

    void stop() {
        if (!mApi) {
            printf("%s: error: trader is not started.\n", mTraderName.c_str());
            return;
        }

        printf("%s: api stopping...\n", mTraderName.c_str());
        mApi->stop();
        delete mApi;
        mApi = nullptr;
    }

    void login() {
        if (!mApi) return;
        printf("%s: api logging in...\n", mTraderName.c_str());
        mApi->login();
    }

    void logout() {
        if (!mApi) return;
        printf("%s: api logging out...\n", mTraderName.c_str());
        mApi->logout();
    }

    void insertOrder(XTFDirection direction,
                     const XTFInstrument *instrument,
                     double price,
                     uint32_t volume) {
        if (!mApi) {
            printf("%s: api is not started.\n", mTraderName.c_str());
            return;
        }

        if (!instrument) {
            printf("%s: instrument invalid.\n", mTraderName.c_str());
            return;
        }

        printf("%s: api prepare order...\n", mTraderName.c_str());
        XTFInputOrder order{};
        order.localOrderID = ++mOrderLocalId; // 建议使用本地唯一的编号
        order.direction = direction;
        order.offsetFlag = XTF_OF_Open;
        order.orderType = XTF_ODT_Limit;
        order.price = price;
        order.volume = volume;
        order.channelSelectionType = XTF_CS_Auto;
        order.channelID = 0;
        order.orderFlag = XTF_ODF_Normal;
        order.instrument = instrument;

        printf("%s: api insert order...\n", mTraderName.c_str());
        mApi->insertOrder(order);
    }

    void buy(const XTFInstrument *instrument, double price, uint32_t volume) {
        insertOrder(XTF_D_Buy, instrument, price, volume);
    }

    void sell(const XTFInstrument *instrument, double price, uint32_t volume) {
        return insertOrder(XTF_D_Sell, instrument, price, volume);
    }

    void cancelAllOrders() {
        if (!mApi) return;
        if (mOrders.empty()) {
            printf("%s: no orders.\n", mTraderName.c_str());
            return;
        }
        for (auto order: mOrders) {
            printf("%s: cancel order: local-id=%d, system-id=%d.\n",
                   mTraderName.c_str(), order.first, order.second);
            mApi->cancelOrder(XTF_OIDT_System, order.second);
        }
    }

    const XTFInstrument* getInstrumentByID(const char *id) {
        if (!mApi) return nullptr;
        return mApi->getInstrumentByID(id);
    }

    void setTraderName(const char *name) {
        mTraderName = name ? name : "nullptr";
    }

private:
    std::map<int, int> mOrders;
    int mOrderLocalId = 0;
    std::string mTraderName;
};

void runExample(const std::string &configPath1,
                const std::string &configPath2,
                const std::string &instrumentId,
                double price,
                uint32_t volume) {
    printf("start example 07.\n");

    Example_07_Trader trader1; {
        trader1.setTraderName("trader-1");
        trader1.setConfigPath(configPath1);
        trader1.start();
        while (!trader1.isStarted())
            trader1.wait(1, "wait for trader-1 started");

        trader1.login();
        while (!trader1.isLoadFinished())
            trader1.wait(1, "wait for trader-1 data load finished");
    }

    Example_07_Trader trader2; {
        trader2.setTraderName("trader-2");
        trader2.setConfigPath(configPath2);
        trader2.start();
        while (!trader2.isStarted())
            trader2.wait(1, "wait for trader-2 started");

        trader2.login();
        while (!trader2.isLoadFinished())
            trader2.wait(1, "wait for trader-2 data load finished");
    }

    auto instrument = trader1.getInstrumentByID(instrumentId.c_str());
    if (instrument) {
        trader2.buy(instrument, price, volume);
        trader2.wait(3, "wait for order inserted");

        trader2.cancelAllOrders(); // no orders.
        trader1.cancelAllOrders(); // cancel success.
        trader1.wait(3, "wait for order canceled");
    }

    trader1.logout();
    while (!trader1.isLoggedOut())
        trader1.wait(1, "wait for trader-1 logout");

    trader1.stop();
    while (!trader1.isStopped())
        trader1.wait(1, "wait for trader-1 stopped");

    trader2.logout();
    while (!trader2.isLoggedOut())
        trader2.wait(1, "wait for trader-2 logout");

    trader2.stop();
    while (!trader2.isStopped())
        trader2.wait(1, "wait for trader-2 stopped");
}

int main(int argc, const char *argv[]) {
    printf("api version: %s.\n", getXTFVersion());

    // TODO: 解析传入参数，提取相关的配置
    std::string configPath1 = "../config-dev/xtf_trader_api_test_156.config";
    std::string configPath2 = "../config-dev/xtf_trader_api_test_156_2.config";
    std::string instrumentId = "au2306";
    double price = 430.50f;
    uint32_t volume = 1;
    runExample(configPath1, configPath2, instrumentId, price, volume);
    return 0;
}
