/**
 * @brief 一个简单的命令行交互模式演示多线程报单
 *
 * 演示功能：
 * 1. 通过配置文件创建API实例；
 * 2. 输入命令调用API接口；
 * 3. 当用户发送报单指令时，启动5个线程发送5个报单；
 */

#include <iostream>
#include <iomanip>
#include <map>
#include "ExampleTrader.h"

#include <thread>
class OrderSender {
public:
    OrderSender() = default;
    ~OrderSender() { stop(); }
    int start() {
        if (!mApi) return -1;
        if (mRunning) return -1;
        mRunning = true;
        mThread = std::thread([&]() {
            while (mRunning) {
                if (mTrigged) {
                    mApi->insertOrder(mInputOrder);
                    mTrigged = false;
                } else {
                    sched_yield();
                }
            }
        });
        return 0;
    }

    int stop() {
        if (!mApi) return -1;
        if (!mRunning) return 0;
        mRunning = false;
        if (mThread.joinable())
            mThread.join();
        return 0;
    }

    int prepareOrder(const XTFInputOrder &inputOrder) {
        if (mTrigged) return -1;
        mInputOrder = inputOrder;
        return 0;
    }

    int trigger() {
        if (mTrigged) return -1;
        mTrigged = true;
        return 0;
    }

    void setTraderAPI(XTFApi *api) { mApi = api; }

private:
    XTFApi *mApi = nullptr;
    volatile bool mRunning = false;
    volatile bool mTrigged = false;
    XTFInputOrder mInputOrder;
    std::thread mThread;
};

class Example_08_Trader : public ExampleTrader {
    OrderSender mOrderSenders[5];

public:
    Example_08_Trader() = default;

    ~Example_08_Trader() override {
        // release api.
        if (mApi) {
            mApi->stop();
            delete mApi;
            mApi = nullptr;
        }
    };

    void onStart(int errorCode, bool isFirstTime) override {
        ExampleTrader::onStart(errorCode, isFirstTime);

        printf("api start ok.\n");
    }

    void onStop(int reason) override {
        ExampleTrader::onStop(reason);
        printf("api stopped, reason: %d.\n", reason);
    }

    void onServerReboot() override {
        printf("trade system reboot, clear local data.\n");
        mInstrument = nullptr;
        mOrders.clear();
    }

    void onLogin(int errorCode, int exchangeCount) override {
        ExampleTrader::onLogin(errorCode, exchangeCount);

        if (errorCode != 0)
            printf("login failed, error code: %d.\n", errorCode);
        else
            printf("login success, exchange count=%d.\n", exchangeCount);
    }

    void onLogout(int errorCode) override {
        ExampleTrader::onLogout(errorCode);

        printf("logout success.\n");
    }

    void onReadyForTrading(const XTFAccount *account) override {
        ExampleTrader::onReadyForTrading(account);

        printf("ready for trading.\n");
        mInstrument = mApi->getInstrumentByID(mInstrumentID.c_str()); // 查询可用的合约对象
        if (!mInstrument) {
            printf("instrument not found: %s\n", mInstrumentID.c_str());
            exit(0);
        }

        mOrderLocalId = account->lastLocalOrderID;
    }

    void onLoadFinished(const XTFAccount *account) override {
        ExampleTrader::onLoadFinished(account);

        printf("load data finished.\n");
    }

    void onAccount(int event, int action, const XTFAccount *account) override {
        // 账户信息发生变化时回调该接口，如：出入金变化
        if (event == XTF_EVT_AccountCashInOut) {
            if (action == XTF_CASH_In) printf("cash in.\n");
            if (action == XTF_CASH_Out) printf("cash out.\n");
        } else {
            printf("account is changed: event=%d, action=%d, accountID=%s\n",
                   event, action, account->accountID);
        }
    }

    void onExchange(int event, int channelID, const XTFExchange *exchange) override {
        // 交易所信息发生变化时回调该接口，如：交易所前置变化
        printf("exchange is changed: event=%d, channelID=%d, exchange=%s\n",
                event, channelID, exchange->exchangeID);
    }

    void onInstrument(int event, const XTFInstrument *instrument) override {
        // 合约属性发生变化时回调该接口，如：状态变化
        if (event == XTF_EVT_InstrumentStatusChanged) {
            printf("instrument status changed: %s %d.\n",
                   instrument->instrumentID, instrument->status);
        }
    }

    void onChangePassword(int errorCode) override {
        printf("password is changed: errorCode=%d\n", errorCode);
    }

    void onOrder(int errorCode, const XTFOrder *order) override {
        printf("recv order report: action=%d, sys-id=%d, "
               "status=%s, error-code=%d.\n",
               order->actionType, order->sysOrderID,
               getOrderStatus(order->orderStatus), errorCode);
        if (errorCode == 0) {
            if (order->orderStatus == XTF_OS_Queuing) {
                mOrders[order->sysOrderID] = order;
            }
        }
    }

    void onCancelOrder(int errorCode, const XTFOrder *cancelOrder) override {
        printf("recv cancel order report: sys-id=%d, status=%s, error-code=%d.\n",
               cancelOrder->sysOrderID, getOrderStatus(cancelOrder->orderStatus), errorCode);
        if (errorCode == 0 || errorCode == 1198) {
            auto iter = mOrders.find(cancelOrder->sysOrderID);
            if (iter != mOrders.end())
                mOrders.erase(iter);
        }
    }

    void onTrade(const XTFTrade *trade) override {
        printf("recv trade report: trade-id=%ld, price=%.4f, volume=%d/%d, sys-order-id=%d\n",
                trade->tradeID, trade->tradePrice, trade->order->totalTradedVolume,
                trade->order->orderVolume, trade->order->sysOrderID);
    }

    void onEvent(const XTFEvent &event) override {
        printf("recv event: %d.\n", event.eventID);
    }

    void onError(int errorCode, void *data, size_t size) override {
        printf("something is wrong, error code: %d.\n", errorCode);
    }

    void start() {
        if (mApi) {
            printf("error: trader has been started.\n");
            return;
        }

        mOrderLocalId = 0;
        mApi = makeXTFApi(mConfigPath.c_str());
        if (mApi == nullptr) {
            printf("error: create xtf api failed, please check config: %s.\n", mConfigPath.c_str());
            exit(0);
        }

        // 重要：多线程报单需要启用此配置选项
        mApi->setConfig("ORDER_MT_ENABLED", "true");

        printf("api starting..., config: %s.\n", mConfigPath.c_str());
        int ret = mApi->start(this);
        if (ret != 0) {
            printf("start failed, error code: %d\n", ret);
            exit(0);
        }

        for (auto &sender: mOrderSenders) {
            sender.setTraderAPI(mApi);
            sender.start();
        }
    }

    void stop() {
        if (!mApi) {
            printf("error: trader is not started.\n");
            return;
        }

        for (auto &sender: mOrderSenders) {
            sender.stop();
        }

        printf("api stopping...\n");
        int ret = mApi->stop();
        if (ret == 0) {
            // API停止操作是异步操作，需要等待一定时间，以防API对象回调时失效。
            // 4.1.664及更高版本不存在此问题，不需要增加延时。
            usleep(100000);

            delete mApi;
            mApi = nullptr;
        } else {
            printf("api stop failed, error code: %d\n", ret);
        }
    }

    void login() {
        if (!mApi) return;
        printf("api logging in...\n");
        int ret = mApi->login();
        if (ret != 0) {
            printf("api logging in failed, error code: %d\n", ret);
        }
    }

    void logout() {
        if (!mApi) return;
        printf("api logging out...\n");
        int ret = mApi->logout();
        if (ret != 0) {
            printf("api logging out failed, error code: %d\n", ret);
        }
    }

    void insertOrder() {
        if (!mApi) {
            printf("api is not started.\n");
            return;
        }

        if (!mInstrument) {
            printf("instrument is not found: %s\n", mInstrumentID.c_str());
            return;
        }

        printf("api prepare order...\n");
        XTFInputOrder order{};
        order.localOrderID = ++mOrderLocalId; // 建议使用本地唯一的编号
        order.direction = XTF_D_Buy; // or XTF_D_Sell
        order.offsetFlag = XTF_OF_Open;
        order.orderType = XTF_ODT_Limit;
        order.price = mPrice;
        order.volume = mVolume;
        order.channelSelectionType = XTF_CS_Auto;
        order.channelID = 0;
        order.orderFlag = XTF_ODF_Normal;
        order.instrument = mInstrument;

#if 1
        // 启动5个线程发送5个报单
        printf("api insert order...\n");
        for (auto &sender: mOrderSenders) sender.prepareOrder(order);
        for (auto &sender: mOrderSenders) sender.trigger();
#else
        // 以循环方式立即发送5个报单
        for (int i = 0; i < 5; ++i) {
            printf("api insert order %d...\n", i+1);
            int ret = mApi->insertOrder(order);
            if (ret != 0) {
                printf("api insert order failed, error code: %d\n", ret);
            }
        }
#endif
    }

    void cancelOrder() {
        if (!mApi) {
            printf("api is not started.\n");
            return;
        }

        if (mOrders.empty()) {
            printf("no orders need cancel.\n");
            return;
        }

        printf("api cancel order...\n");
        auto orders = mOrders;
        printf("order count: %lu\n", orders.size());
        for (auto &iter: orders) {
            printf("cancel order: sys-id=%d.\n", iter.first);
            int ret = mApi->cancelOrder(iter.second);
            if (ret != 0) {
                printf("api cancel order failed, error code: %d\n", ret);
            }
            usleep(500000); // sleep 500ms;
        }
    }

    void printAccount() {
        auto account = mApi->getAccount();
        std::cout << "accountID=" << account->accountID << std::endl;
        std::cout << "preBalance=" << account->preBalance << std::endl;
        std::cout << "staticBalance=" << account->staticBalance << std::endl;
        std::cout << "deliveryMargin=" << account->deliveryMargin << std::endl;
        std::cout << "deposit=" << account->deposit << std::endl;
        std::cout << "withdraw=" << account->withdraw << std::endl;
        std::cout << "margin=" << account->margin << std::endl;
        std::cout << "frozenMargin=" << account->frozenMargin << std::endl;
        std::cout << "premium=" << account->premium << std::endl;
        std::cout << "frozenPremium=" << account->frozenPremium << std::endl;
        std::cout << "commission=" << account->premium << std::endl;
        std::cout << "frozenCommission=" << account->frozenCommission << std::endl;
        std::cout << "balance=" << account->balance << std::endl;
        std::cout << "available=" << account->available << std::endl;
        std::cout << "positionProfit=" << account->positionProfit << std::endl;
        std::cout << "closeProfit=" << account->closeProfit << std::endl;
    }

    void printOrders() {
        auto account = mApi->getAccount();
        int orderTotalCount = account->getOrderCount();
        if (orderTotalCount == 0) {
            std::cout << "no orders." << std::endl;
            return;
        }

        auto printOrderDetail = [](const XTFOrder &order) {
            std::cout << "order(" << (int) order.orderType;
            std::cout << "): direction=" << (int) order.direction;
            std::cout << ", offsetFlag=" << (int) order.offsetFlag;
            std::cout << ", instrumentID=" << (order.instrument ? order.instrument->instrumentID : "unknown");
            std::cout << ", localOrderID=" << order.localOrderID;
            std::cout << ", sysOrderID=" << order.sysOrderID;
            std::cout << ", exchangeOrderID=" << order.exchangeOrderID;
            std::cout << ", price=" << order.orderPrice;
            std::cout << ", volume=" << order.totalTradedVolume << "/" << order.orderVolume;
            std::cout << ", minVolume=" << order.orderMinVolume;
            std::cout << ", status=" << (int) order.orderStatus;
            if (order.isHistory) std::cout << ", history";
            std::cout << ", insertTime=" << order.insertTime;
            if (order.orderStatus == XTF_OS_Canceled)
                std::cout << ", cancelTime=" << order.cancelTime;
            else
                std::cout << ", updateTime=" << order.updateTime;
            std::cout << std::endl;
        };

        int orderCount = 0;
        const XTFOrder *order;
        for (int i = 0; i < orderTotalCount; ++i) {
            order = account->getOrder(i);
            if (!order) continue;
            ++orderCount;
            std::cout << orderCount << ": ";
            printOrderDetail(*order);
        }

        std::cout << "orders total count: " << std::to_string(orderCount) << std::endl;
    }

    void printTrades() {
        auto account = mApi->getAccount();
        int tradeCount = account->getTradeCount();
        if (tradeCount == 0) {
            std::cout << "no trades." << std::endl;
            return;
        }

        auto printTradeDetail = [](const XTFTrade &trade) {
            std::cout << "trade(direction=" << (int) trade.direction
                      << ", offsetFlag=" << (int) trade.offsetFlag
                      << ") tradeID=" << trade.tradeID
                      << ", price=" << trade.tradePrice
                      << ", volume=" << trade.tradeVolume
                      << ", sysOrderID=" << (trade.order ? trade.order->sysOrderID : -1)
                      << ", instrumentID=" << (trade.order ? trade.order->instrument->instrumentID : "unknown")
                      << (trade.isHistory ? ", history" : "")
                      << ", tradeTime=" << trade.tradeTime;
            if (trade.isSelfTraded()) std::cout << ", selfTraded";
            std::cout << std::endl;
        };

        const XTFTrade *trade;
        for (int i = 0; i < tradeCount; ++i) {
            trade = account->getTrade(i);
            printTradeDetail(*trade);
        }
        std::cout << "trades total count: " << std::to_string(tradeCount) << std::endl;
    }

    void printPositions() {
        auto account = mApi->getAccount();
        int positionCount = account->getPositionCount();
        if (positionCount == 0) {
            std::cout << "no positions." << std::endl;
            return;
        }

        std::multimap<std::string, const XTFPosition *> positions;
        for (int i = 0; i < positionCount; ++i) {
            const XTFPosition* position = account->getPosition(i);
            if (!position) continue;
            positions.emplace(position->getInstrument()->instrumentID, position);
        }

        auto printPositionDetails = [](const XTFPosition &position) {
            std::cout << "position: " << position.getInstrument()->instrumentID
                      << ", direction=" << (int) position.direction
                      << ", position=" << position.position
                      << ", todayPosition=" << position.todayPosition
                      << ", yesterdayPosition=" << position.getYesterdayPosition()
                      << ", availablePosition=" << position.getAvailablePosition()
                      << ", combinedPosition=" << position.combinedPosition
                      << ", openPrice=" << position.getOpenPrice()
                      << ", openFrozen=" << position.openFrozen
                      << ", closeFrozen=" << position.closeFrozen
                      << ", margin=" << position.margin
                      << ", paidMargin=" << position.paidMargin
                      << ", frozenMargin=" << position.frozenMargin
                      << ", frozenCommission=" << position.frozenCommission
                      << ", positionProfit=" << position.positionProfit
                      << ", closeProfit=" << position.closeProfit
                      << std::endl;

            int positionDetailCount = position.getPositionDetailCount();
            std::cout << "details: " << positionDetailCount << std::endl;
            for (int i = 0; i < position.getPositionDetailCount(); ++i) {
                const XTFPositionDetail &detail = position.getPositionDetail(i);
                std::cout << "  openPrice=" << detail.openPrice
                          << ", openVolume=" << detail.openVolume
                          << ", remainingVolume=" << detail.remainingVolume
                          << ", combinedVolume=" << detail.getCombinedVolume()
                          << ", history=" << (detail.isHistory() ? "true" : "false")
                          << ", openTradeID=" << detail.openTradeID
                          << ", close trade list: " << detail.getCloseTradeCount()
                          << ", combDetails: " << detail.getCombPositionDetailCount()
                          << std::endl;

                int closeTradeCount = detail.getCloseTradeCount();
                if (closeTradeCount > 0) {
                    std::cout << "  close trade list:" << std::endl;
                    for (int j = 0; j < detail.getCloseTradeCount(); ++j) {
                        const XTFTradeDetail &trade = detail.getCloseTradeDetail(j);
                        std::cout << "  "
                                  << "  tradeID=" << trade.trade->tradeID
                                  << ", tradeVolume=" << trade.volume
                                  << "/" << trade.trade->tradeVolume
                                  << ", tradePrice=" << trade.trade->tradePrice
                                  << std::endl;
                    }
                }

                int combDetailCount = detail.getCombPositionDetailCount();
                if (combDetailCount > 0) {
                    std::cout << "  comb-details:" << std::endl;
                    for (int j = 0; j < detail.getCombPositionDetailCount(); ++j) {
                        const XTFCombPositionDetail &combDetail = detail.getCombPositionDetail(j);
                        std::cout << "  "
                                  << "  volume=" << combDetail.volume
                                  << ", leftTradeID=" << combDetail.leftTradeID
                                  << ", rightTradeID=" << combDetail.rightTradeID
                                  << std::endl;
                    }
                }
            }
        };

        for (auto &item: positions) {
            printPositionDetails(*item.second);
        }
        std::cout << "positions total count: " << std::to_string(positionCount) << std::endl;
    }

private:
    const XTFInstrument *mInstrument;
    std::map<int, const XTFOrder *> mOrders;
    int mOrderLocalId;
};

void runExample(const std::string &configPath, const std::string &instrumentId, double price, int volume) {
    printf("start example 08.\n");

    Example_08_Trader trader;
    trader.setConfigPath(configPath);
    trader.setInstrumentID(instrumentId);
    trader.setPrice(price);
    trader.setVolume(volume);

    printf("api version: %s\n", getXTFVersion());
    printf("input 'quit' or 'exit' to exit.\n");
    std::string cmdline;
    std::string prefix = "> ";
    while (true) {
        std::cout << prefix;
        std::getline(std::cin, cmdline);
        if (cmdline.empty()) continue;
        if (cmdline == "start") {
            if (trader.isStarted()) {
                printf("trader is started.\n");
            } else {
                trader.start();
            }
            continue;
        }
        if (cmdline == "stop") {
            if (trader.isStopped()) {
                printf(".\n");
            } else {
                trader.stop();
            }
            continue;
        }
        if (cmdline == "login") {
            if (trader.isLoggedIn()) {
                printf("trader is logged in.\n");
            } else {
                trader.login();
            }
            continue;
        }
        if (cmdline == "logout") {
            if (trader.isLoggedOut()) {
                printf("trader is logged out.\n");
            } else {
                trader.logout();
            }
            continue;
        }
        if (cmdline == "insert") {
            if (!trader.isLoadFinished()) {
                printf("wait for data load finished.\n");
            } else {
                trader.insertOrder();
            }
            continue;
        }
        if (cmdline == "cancel") {
            if (!trader.isLoadFinished()) {
                printf("wait for data load finished.\n");
            } else {
                trader.cancelOrder();
            }
            continue;
        }
        if (cmdline == "orders") {
            if (!trader.isLoadFinished()) {
                printf("wait for data load finished.\n");
            } else {
                trader.printOrders();
            }
            continue;
        }
        if (cmdline == "trades") {
            if (!trader.isLoadFinished()) {
                printf("wait for data load finished.\n");
            } else {
                trader.printTrades();
            }
            continue;
        }
        if (cmdline == "positions") {
            if (!trader.isLoadFinished()) {
                printf("wait for data load finished.\n");
            } else {
                trader.printPositions();
            }
            continue;
        }
        if (cmdline == "account") {
            if (!trader.isLoadFinished()) {
                printf("wait for data load finished.\n");
            } else {
                trader.printAccount();
            }
            continue;
        }

        if (cmdline == "exit" || cmdline == "quit") {
            break;
        } else {
            printf("start     start trader.\n");
            printf("stop      stop trader.\n");
            printf("login     login trader.\n");
            printf("logout    logout trader.\n");
            printf("insert    insert default order.\n");
            printf("cancel    cancel existed orders.\n");
            printf("orders    print orders.\n");
            printf("trades    print trades.\n");
            printf("positions print positions.\n");
            printf("account   print account info.\n");
            printf("exit      exit application.\n");
            printf("help      show help.\n");
            continue;
        }
    }
}

int main(int argc, const char *argv[]) {
    std::cout << std::fixed << std::setprecision(6);
    // TODO: 解析传入参数，提取相关的配置
    //std::string configPath = "../config/xtf_trader_api.config";
    std::string configPath = "../config-dev/xtf_trader_api_test_156.config";
    std::string instrumentId = "au2308";
    double price = 301.50;
    int volume = 1;
    runExample(configPath, instrumentId, price, volume);
    return 0;
}
