/**
 * @brief 一个简单的协议报撤行权对冲单功能演示
 *
 * 演示功能：
 * 1. 通过配置文件创建API实例；
 * 2. 启动API，启动成功后，调用login接口登录柜台；
 * 3. 等数据加载完毕后，使用协议方式发送行权报单请求；
 * 4. 等待3秒后，针对所有的行权报单，使用协议方式尝试撤单；
 * 5. 等待3秒后，调用logout接口登出柜台；
 */

#include <fcntl.h>
#include <unistd.h>
#if defined(_WIN32) || defined(WIN32)
#include <winsock2.h>
#include <ws2tcpip.h>
#else
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <arpa/inet.h>
#endif

#include <map>
#include "ExampleTrader.h"

#pragma pack(push, 1)

/// 行权单报文消息: 行权/放弃行权/自对冲插入请求
struct CXeleFairInputExecOrderMsg {
    uint8_t                 messageID;                      ///< 消息id, 报单操作为107
    uint8_t                 clientIndex;                    ///< 客户端index
    uint32_t                token;                          ///< 客户端token
    int                     seqNo;                          ///< 消息序列号
    uint32_t                requestID;                      ///< 请求id
    int                     localOrderID;                   ///< 报单本地id
    char                    instrumentID[16];               ///< 合约代码
    uint32_t                instrumentIndex;                ///< 合约序号
    short                   volume;                         ///< 数量
    double                  minProfit;                      ///< 最小盈利(广期所需要，暂未使用)
    uint8_t                 execFlag;                       ///< 行权/自对冲标记
    char                    offsetFlag;                     ///< 开平标志(只有上期所/能源所需要)
    char                    hedgeFlag;                      ///< 投机套保标记类型
    uint8_t                 closeFlag;                      ///< 自对冲标记类型
    uint8_t                 exchangeFront;                  ///< 前置信息 不指定前置填写为0，指定前置需要加上偏移量10，如指定前置3，则需要填写13
    int32_t                 userRef;                        ///< 用户自定义数据
    char                    reserve[7];                     ///< 预留字段
};
static_assert(sizeof(CXeleFairInputExecOrderMsg) == 64, "CXeleFairInputExecOrderMsg size error");

/// 行权单报文消息: 行权/放弃行权/自对冲撤销请求
struct CXeleFairExecOrderActionMsg{
    uint8_t                 messageID;                      ///< 消息id, 撤单操作为109
    uint8_t                 clientIndex;                    ///< 客户端index
    uint32_t                token;                          ///< 客户端token
    int                     seqNo;                          ///< 消息序列号
    uint32_t                requestID;                      ///< 请求id
    int                     localActionID;                  ///< 本地操作编号
    int                     systemOrderID;                  ///< 被撤单柜台编码
    char                    instrumentID[16];               ///< 合约代码
    uint32_t                instrumentIndex;                ///< 合约序号
    uint8_t                 execFlag;                       ///< 行权/自对冲标记
    char                    actionFlag;                     ///< 报单操作标志
    int                     localOrderID;                   ///< 本地报单编号
    uint8_t                 exchangeFront;                  ///< 前置信息 不指定前置填写为0，指定前置需要加上偏移量10，如指定前置3，则需要填写13
    int32_t                 userRef;                        ///< 用户自定义数据
    char                    reserve[11];                    ///< 预留字段
};
static_assert(sizeof(CXeleFairExecOrderActionMsg) == 64, "CXeleFairExecOrderActionMsg size error");

#pragma pack(pop)


class Example_09_Trader : public ExampleTrader {
public:
    Example_09_Trader() = default;
    ~Example_09_Trader() = default;

    void onStart(int errorCode, bool isFirstTime) override {
        ExampleTrader::onStart(errorCode, isFirstTime);

        printf("api start ok.\n");
    }

    void onStop(int reason) override {
        ExampleTrader::onStop(reason);

        printf("api stopped, reason: %d.\n", reason);
    }

    void onServerReboot() override {
        printf("trade system reboot, clear local data.\n");
        mInstrument = nullptr;
        mOrders.clear();
    }

    void onLogin(int errorCode, int exchangeCount) override {
        ExampleTrader::onLogin(errorCode, exchangeCount);

        if (errorCode != 0) {
            printf("login failed, error code: %d.\n", errorCode);
        } else {
            printf("login success, exchange count=%d.\n", exchangeCount);
        }
    }

    void onLogout(int errorCode) override {
        ExampleTrader::onLogout(errorCode);

        printf("logout success.\n");
    }

    void onReadyForTrading(const XTFAccount *account) override {
        ExampleTrader::onReadyForTrading(account);

        printf("ready for trading.\n");
        mInstrument = mApi->getInstrumentByID(mInstrumentID.c_str()); // 查询可用的合约对象
        if (!mInstrument) {
            printf("instrument not found: %s\n", mInstrumentID.c_str());
            exit(0);
        }

        mExchange = mInstrument->getProduct()->getProductGroup()->getExchange();
        if (!mExchange) {
            printf("instrument error, exchange invalid: %s\n", mInstrumentID.c_str());
            exit(0);
        }

        mLocalOrderID = account->lastLocalOrderID;   // API&协议报单使用
        mLocalActionID = account->lastLocalActionID; // 协议报单使用
    }

    void onLoadFinished(const XTFAccount *account) override {
        ExampleTrader::onLoadFinished(account);

        printf("load data finished.\n");
    }

    void onAccount(int event, int action, const XTFAccount *account) override {
        // 账户信息发生变化时回调该接口，如：出入金变化
        if (event == XTF_EVT_AccountCashInOut) {
            if (action == XTF_CASH_In) printf("cash in.\n");
            if (action == XTF_CASH_Out) printf("cash out.\n");
        } else {
            printf("account is changed: event=%d, action=%d, accountID=%s\n",
                   event, action, account->accountID);
        }
    }

    void onExchange(int event, int channelID, const XTFExchange *exchange) override {
        // 交易所信息发生变化时回调该接口，如：交易所前置变化
        printf("exchange is changed: event=%d, channelID=%d, exchange=%s\n",
               event, channelID, exchange->exchangeID);
    }

    void onInstrument(int event, const XTFInstrument *instrument) override {
        // 合约属性发生变化时回调该接口，如：状态变化
        if (event == XTF_EVT_InstrumentStatusChanged) {
            printf("instrument status changed: %s %d.\n",
                   instrument->instrumentID, instrument->status);
        }
    }

    void onChangePassword(int errorCode) override {
        printf("password is changed: errorCode=%d\n", errorCode);
    }

    void onExecOrder(int errorCode, const XTFOrder *order) override {
        printf("recv exec order report: action=%d, sys-id=%d, "
               "status=%s, error-code=%d.\n",
               order->actionType, order->sysOrderID,
               getOrderStatus(order->orderStatus), errorCode);
        if (errorCode == 0) {
            if (order->orderStatus == XTF_OS_Queuing) {
                mOrders[order->sysOrderID] = order;
            }
        }
    }

    void onCancelExecOrder(int errorCode, const XTFOrder *cancelOrder) override {
        printf("recv cancel exec order report: sys-id=%d, status=%s, error-code=%d.\n",
               cancelOrder->sysOrderID, getOrderStatus(cancelOrder->orderStatus), errorCode);
        if (errorCode == 0 || errorCode == 1198) { // 1198: 报单已经撤销或成交或为错单
            auto iter = mOrders.find(cancelOrder->sysOrderID);
            if (iter != mOrders.end())
                mOrders.erase(iter);
        }
    }

    void onTrade(const XTFTrade *trade) override {
        printf("recv trade report: trade-id=%ld, price=%.4f, volume=%d/%d, sys-order-id=%d\n",
                trade->tradeID, trade->tradePrice, trade->order->totalTradedVolume,
                trade->order->orderVolume, trade->order->sysOrderID);
    }

    void onEvent(const XTFEvent &event) override {
        printf("recv event: %d.\n", event.eventID);
    }

    void onError(int errorCode, void *data, size_t size) override {
        printf("something is wrong, error code: %d.\n", errorCode);
    }

    void start() {
        if (mApi) {
            printf("error: trader has been started.\n");
            return;
        }

        mLocalOrderID = 0;
        mApi = makeXTFApi(mConfigPath.c_str());
        if (mApi == nullptr) {
            printf("error: create xtf api failed, please check config: %s.\n", mConfigPath.c_str());
            exit(0);
        }

        // 保存报单的地址和端口，用于协议方式发送报撤单。
        mTradeServerIp = mApi->getConfig("TRADE_SERVER_IP");
        mTradeServerPort = mApi->getConfig("TRADE_SERVER_PORT");
        openUdpSocket();

        printf("api starting..., config: %s.\n", mConfigPath.c_str());
        int ret = mApi->start(this);
        if (ret != 0) {
            printf("start failed, error code: %d\n", ret);
            exit(0);
        }
    }

    void stop() {
        if (!mApi) {
            printf("error: trader is not started.\n");
            return;
        }

        closeUdpSocket();

        printf("api stopping...\n");
        int ret = mApi->stop();
        if (ret == 0) {
            // API停止操作是异步操作，需要等待一定时间，以防API对象回调时失效。
            // 4.1.664及更高版本不存在此问题，不需要增加延时。
            usleep(100000);

            delete mApi;
            mApi = nullptr;
        } else {
            printf("api stop failed, error code: %d\n", ret);
        }
    }

    void login() {
        if (!mApi) return;
        printf("api logging in...\n");
        int ret = mApi->login();
        if (ret != 0) {
            printf("api logging in failed, error code: %d\n", ret);
        }
    }

    void logout() {
        if (!mApi) return;
        printf("api logging out...\n");
        int ret = mApi->logout();
        if (ret != 0) {
            printf("api logging out failed, error code: %d\n", ret);
        }
    }

    // 协议方式报单
    void insertExecOrderByUdp() {
        if (!mApi) {
            printf("api is not started.\n");
            return;
        }

        if (!mInstrument) {
            printf("instrument is not found: %s\n", mInstrumentID.c_str());
            return;
        }

        if (!mExchange) {
            printf("exchange invalid\n");
            return;
        }

        printf("udp prepare exec order...\n");
        CXeleFairInputExecOrderMsg order{};
        order.messageID = 0x6b;
        order.clientIndex = mExchange->clientIndex;
        order.token = mExchange->clientToken;
        order.localOrderID = ++mLocalOrderID;
        order.requestID = ++mRequestID;
        order.seqNo = ++mSeqNo;
        strcpy(order.instrumentID, mInstrument->instrumentID);
        order.instrumentIndex = mInstrument->instrumentIndex;
        order.volume = static_cast<short>(mVolume);
        order.minProfit = 10.0;
        order.execFlag = 1; // 行权
        order.hedgeFlag = '1'; // 投机
        order.offsetFlag = '1'; // 开仓
        order.closeFlag = 1; // 对冲
        order.exchangeFront = 0; // 自动选择
        order.userRef = 0;

        printf("udp insert exec order...\n");
        sendUdpData(&order, sizeof(order));
    }

    // 协议方式撤单
    void cancelExecOrderByUdp() {
        if (!mApi) {
            printf("api is not started.\n");
            return;
        }

        if (mOrders.empty()) {
            printf("no orders need cancel.\n");
            return;
        }

        if (!mExchange) {
            printf("exchange invalid\n");
            return;
        }

        printf("udp cancel exec order...\n");
        auto orders = mOrders;
        printf("exec order count: %lu\n", orders.size());
        for (auto &iter: orders) {
            printf("cancel exec order: sys-id=%d.\n", iter.first);
            CXeleFairExecOrderActionMsg orderAction{};
            orderAction.messageID = 0x6d;
            orderAction.clientIndex = mExchange->clientIndex;
            orderAction.token = mExchange->clientToken;
            orderAction.seqNo = ++mSeqNo;
            orderAction.requestID = ++mRequestID;
            orderAction.localActionID = ++mLocalActionID;
            orderAction.systemOrderID = iter.first;
            strcpy(orderAction.instrumentID, iter.second->instrument->instrumentID);
            orderAction.instrumentIndex = iter.second->instrument->instrumentIndex;
            orderAction.execFlag = 1;
            orderAction.actionFlag = '0';
            orderAction.localOrderID = iter.second->localOrderID;
            orderAction.exchangeFront = 0;
            orderAction.userRef = 0;
            sendUdpData(&orderAction, sizeof(orderAction));
            usleep(500000); // sleep 500ms;
        }
    }

    void openUdpSocket() {
        if (mUdpSocket != -1) return;
        mUdpSocket = ::socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
        if (mUdpSocket == -1) {
            printf("udp create socket failed: %d\n", errno);
            exit(1);
        }
        printf("udp channel create socket success: %d\n", mUdpSocket);

        std::string remoteIp = mTradeServerIp;
        uint16_t remotePort = (uint16_t) std::stoi(mTradeServerPort);
        struct sockaddr_in sa{};
        memset(&sa, 0, sizeof(sa));
        sa.sin_family = AF_INET;
        sa.sin_addr.s_addr = inet_addr(remoteIp.c_str());
        sa.sin_port = htons(remotePort);

        int result = connect(mUdpSocket, (struct sockaddr *) &sa, sizeof(sockaddr_in));
        if (result == -1) {
            printf("udp channel connect[%s:%u] failed, errno: %d.", remoteIp.c_str(), remotePort, errno);
            ::close(mUdpSocket); // 释放创建的套接字
            mUdpSocket = -1;
            exit(2);
        }
    }

    void closeUdpSocket() {
        if (mUdpSocket == -1) return;
        ::close(mUdpSocket);
        mUdpSocket = -1;
    }

    void sendUdpData(const void *data, size_t len) {
#if defined(_WIN32) || defined(WIN32)
        int result = ::send(mUdpSocket, (const char *)data, len, 0);
#else
        int result = ::send(mUdpSocket, data, len, 0);
#endif
        if (result == (int) len) {
            printf("udp send ok, data size: %d\n", result);
        } else {
            printf("udp send error: %d\n", result);
        }
    }

private:
    const XTFExchange *mExchange;
    const XTFInstrument *mInstrument;
    std::map<int, const XTFOrder *> mOrders;
    int mLocalOrderID = 0;
    int mLocalActionID = 0;
    int mUdpSocket = -1;
    std::string mTradeServerIp;
    std::string mTradeServerPort;
    int mSeqNo = 0;
    uint32_t mRequestID = 0;
};


void runExample(const std::string &configPath, const std::string &instrumentId, double price, int volume) {
    printf("start example 09.\n");

    Example_09_Trader trader;
    trader.setConfigPath(configPath);
    trader.setInstrumentID(instrumentId);
    trader.setPrice(price);
    trader.setVolume(volume);

    trader.start();
    while (!trader.isStarted())
        trader.wait(1, "wait for trader started");

    trader.login();
    while (!trader.isLoadFinished())
        trader.wait(1, "wait for data load finished");

    trader.insertExecOrderByUdp();
    trader.wait(3, "wait for exec order inserted");

    trader.cancelExecOrderByUdp();
    trader.wait(3, "wait for exec order canceled");

    trader.logout();
    while (!trader.isLoggedOut())
        trader.wait(1, "wait for trader logout");

    trader.stop();
    while (!trader.isStopped())
        trader.wait(1, "wait for trader stopped");
}

int main(int argc, const char *argv[]) {
    printf("api version: %s.\n", getXTFVersion());

    // TODO: 解析传入参数，提取相关的配置
    std::string configPath = "../config/xtf_trader_api.config";
    std::string instrumentId = "zn2309C8300";
    double price = 8000.0;
    int volume = 1;
    runExample(configPath, instrumentId, price, volume);
    return 0;
}
