/**
 * @file          XTFApi.h
 * @brief         API接口头文件
 * @details       本文档主要给出了API接口的详细说明和定义
 * @author        xinweiz
 * @date          2022-06-22
 * @version       v4.1
 * @copyright     南京艾科朗克信息科技有限公司
 */

#ifndef XTF_API_H
#define XTF_API_H

#include <unistd.h>

#include "XTFDataStruct.h"
#define XTF_API_VERSION "4.1.1375-c0fd85f"

class XTF_API_EXPORT XTFSpi {
public:
    virtual ~XTFSpi() = default;

    virtual const char * apiVersion() { return XTF_API_VERSION; }

    ///////////////////////////// 生命期管理接口 ////////////////////////////////

    /**
     * API对象启动通知接口
     *
     * 如果API启动后发生连接中断，API会根据配置发起重新连接。
     * 连接成功后，API重新调用onStart接口通知用户，此时isFirstTime为false。
     *
     * 如果API登录后发生连接中断，API重连成功后并不会发起登录请求，需要用户主动发起登录操作。
     *
     * @param errorCode     启动结果。
     *                      0-表示启动成功，可以调用 login() 接口登录柜台；
     *                      X-表示启动失败对应的错误码；
     *
     * @param isFirstTime   是否初次打开。
     *                      可以据此标志位，判断在初次打开时，进行数据的初始化等操作。
     */
    virtual void onStart(int errorCode, bool isFirstTime) {}

    /**
     * API对象停止通知接口
     *
     * @param reason        停止的原因。
     * - ERRXTFAPI_ClosedByUser          : 客户调用stop接口主动关闭
     * - ERRXTFAPI_ClosedByTimeout       : 心跳超时主动关闭
     * - ERRXTFAPI_ClosedBySendError     : 发送数据时，检测到套接字异常，主动关闭
     * - ERRXTFAPI_ClosedByRecvError     : 检测到对端关闭TCP连接
     * - ERRXTFAPI_ClosedByLoginError    : 检测到登录应答协议异常后主动关闭
     * - ERRXTFAPI_ClosedByLogoutError   : 检测到登出应答协议异常后主动关闭
     * - ERRXTFAPI_ClosedByZipStreamError: 检测到流水推送数据异常后主动关闭
     */
    virtual void onStop(int reason) {}

    /**
     * 柜台在交易时段发生重启的通知接口
     *
     * 说明：
     * 1. 柜台在交易时段如果发生重启，API中断后会自动重连；
     * 2. 重新连接后，API会清空上一次登录的所有数据，并重新从柜台加载数据；
     * 3. 由于API的本地数据发生了变化，因此所有外部使用的指针数据会失效，用户需要在收到
     *    onServerReboot事件后，清理上一次的所有数据指针，此刻这些数据指针依然有效；
     *    当onServerReboot事件处理之后，数据指针将会失效。
     */
    virtual void onServerReboot() {}



    ///////////////////////////// 会话管理接口 ////////////////////////////////

    /**
     * 登录结果通知接口
     *
     * @param errorCode     登录结果。
     *                      0-表示登录成功；
     *                      非0-表示发生错误的错误码；
     *                      具体原因请查阅API说明文档或者官方文档平台(http://docs.accelecom.com/)
     *                      关于期货柜台的错误码说明
     *
     * @param exchangeCount 交易所数量，登录成功后返回交易所的可用数量，但此时不能查询交易席位信息。
     *                      需要收到 onReadyForTrading() 或 onLoadFinished() 事件通知时，
     *                      才能查询交易所的交易席位信息。
     */
    virtual void onLogin(int errorCode, int exchangeCount) {}

    /**
     * 登出结果通知接口
     *
     * @param errorCode     登出结果。
     *                      0-表示登出成功；
     *                      非0-表示发生错误的错误码；
     *                      具体原因请查阅API说明文档或者官方文档平台(http://docs.accelecom.com/)
     *                      关于期货柜台的错误码说明
     */
    virtual void onLogout(int errorCode) {}

    /**
     * 修改密码结果通知接口
     *
     * @param errorCode     修改结果。
     *                      0-表示修改成功；
     *                      非0-表示发生错误的错误码；
     *                      具体原因请查阅API说明文档或者官方文档平台(http://docs.accelecom.com/)
     *                      关于期货柜台的错误码说明
     */
    virtual void onChangePassword(int errorCode) {}

    /**
     * 日初静态数据已加载完毕，可进行报单操作。
     *
     * 日初静态数据已经加载完毕，用户可以根据需要查询合约对象，并进行报单；
     * 但是，报单回报会在日内历史流水追平之后，才会到达。
     *
     * 注意，至此阶段日内历史流水数据并未加载完成，暂时无法计算资金和仓位，
     * 如果需要根据资金和仓位进行报单，需要等待流水追平之后，再发送报单。
     *
     * @param account       登录用户账号对象
     *                      在API实例生命周期内，此对象指针一直有效，可以保存使用；
     *                      该对象指针与getAccount()接口查询的一致；
     */
    virtual void onReadyForTrading(const XTFAccount *account) {}

    /**
     * 交易日内所有流水数据已加载完毕
     *
     * 用户可以根据流水数据计算资金和仓位，并据此进行报单。
     *
     * @param account       登录用户账号对象
     *                      在API实例生命周期内，此对象指针一直有效，可以保存使用；
     *                      该对象指针与getAccount()接口查询的一致；
     */
    virtual void onLoadFinished(const XTFAccount *account) {}



    //////////////////////////////// 交易接口 ////////////////////////////////

    /**
     * 报单回报及订单状态改变通知接口
     *
     * IOC订单不成交被交易所撤单时不产生该事件，只产生onCancelOrder事件；成交时会产生onOrder事件和onTrade事件。
     *
     * 说明：
     * 1、该事件在流水重传的时候也会产生,此时isHistory为true;
     * 2、流水数据支持断点续传功能；
     * 3、如果是新创建的API对象，流水数据从头开始推送；
     * 4、如果API对象已存在，连接断开后重连，流水数据从断开处续传推送；
     * @param errorCode     操作错误码
     *                      0-表示成功；非0-表示发生错误的错误码；
     *                      具体原因请查阅API说明文档或者官方文档平台(http://docs.accelecom.com/)
     *                      关于期货柜台的错误码说明
     * @param order         报单回报对象
     *                      XTFOrder对象被创建后，在API实例的生命周期内是一直有效的。
     *                      但是，已成交数量、报单状态等部分字段在接收不同阶段onOrder事件时会更新为不同的值。
     *                      如果需要对报单进行异步处理，那么需要将XTFOrder的字段拷贝到用户管理的内存之后，再做下一步处理。
     *                      否则，XTFOrder对象的可变字段可能被下一次订单状态更新而刷新。
     *
     * 关于订单状态变化的补充说明：
     * 由于交易柜台支持主席平仓的功能，从管理席位接收回报，所以生产环境下会出现订单状态乱序的情况。
     * API如果先收到 (XTF_OS_Queuing, XTF_OS_Canceled, XTF_OS_PartTraded, XTF_OS_AllTraded, XTF_OS_Rejected)
     * 这几种状态中的一种，后收到 XTF_OS_Accepted 状态，将不产生XTF_OS_Accepted状态的事件。
     *
     */
    virtual void onOrder(int errorCode, const XTFOrder *order) {}


    /**
     * 撤单回报通知接口
     *
     * 主动撤单成功或者失败、IOC订单被交易所撤单都会产生该事件，撤单成功时该订单orderStatus为XTF_OS_Canceled
     *
     * 说明：
     * 1、该事件在流水重传的时候也会产生，此时isHistory为true;
     * 2、流水数据支持断点续传功能；
     * 3、如果是新创建的API对象，流水数据从头开始推送；
     * 4、如果API对象已存在，连接断开后重连，流水数据从断开处续传推送；
     *
     * @param errorCode     操作错误码
     *                      0-表示成功；非0-表示发生错误的错误码；
     *                      具体原因请查阅API说明文档或者官方文档平台(http://docs.accelecom.com/)
     *                      关于期货柜台的错误码说明
     * @param cancelOrder   撤单回报对象，同onOrder接口说明
     *                      如果为nullptr，则表示一键多条件撤单发生错误时的回调，查看errorCode来确认错误的原因
     */
    virtual void onCancelOrder(int errorCode, const XTFOrder *cancelOrder) {}

    /**
     * 成交回报通知接口
     *
     * 说明：
     * 1、该事件在流水重传的时候也会产生,此时isHistory为true;
     * 2、流水数据支持断点续传功能；
     * 3、如果是新创建的API对象，流水数据从头开始推送；
     * 4、如果API对象已存在，连接断开后重连，流水数据从断开处续传推送；
     *
     * @param trade         成交回报对象
     *                      XTFTrade对象被创建后，在API实例的生命周期内是一直有效的。
     *                      虽然XTFTrade对象的字段不再更新，但是关联的XTFOrder对象和成交对应的仓位明细会发生变化。
     *                      如果需要对成交回报进行异步处理，建议将XTFTrade的字段拷贝到用户管理的内存之后，再做下一步处理。
     *                      如果用户不关心报单状态、已成交数量及成交明细，那么也可以直接使用XTFTrade指针。
     */
    virtual void onTrade(const XTFTrade *trade) {}

    /**
     * 行权或对冲报单回报通知接口
     *
     * 说明：
     * 1、该事件在流水重传的时候也会产生,此时isHistory为true;
     * 2、流水数据支持断点续传功能；
     * 3、如果是新创建的API对象，流水数据从头开始推送；
     * 4、如果API对象已存在，连接断开后重连，流水数据从断开处续传推送；
     * @param errorCode     操作错误码
     *                      0-表示成功；非0-表示发生错误的错误码；
     *                      具体原因请查阅API说明文档或者官方文档平台(http://docs.accelecom.com/)
     *                      关于期货柜台的错误码说明
     * @param order         报单回报对象
     *                      XTFOrder对象被创建后，在API实例的生命周期内是一直有效的。
     *
     * 关于订单状态变化的补充说明：
     * 行权或对冲报单回报，没有以下状态：
     * - XTF_OS_PartTraded
     * - XTF_OS_AllTraded
     */
    virtual void onExecOrder(int errorCode, const XTFOrder *order) {}


    /**
     * 行权或对冲撤单回报通知接口
     *
     * 主动撤单成功或者失败会产生该事件，撤单成功时该订单orderStatus为XTF_OS_Canceled
     *
     * 说明：
     * 1、该事件在流水重传的时候也会产生，此时isHistory为true;
     * 2、流水数据支持断点续传功能；
     * 3、如果是新创建的API对象，流水数据从头开始推送；
     * 4、如果API对象已存在，连接断开后重连，流水数据从断开处续传推送；
     *
     * @param errorCode     操作错误码
     *                      0-表示成功；非0-表示发生错误的错误码；
     *                      具体原因请查阅API说明文档或者官方文档平台(http://docs.accelecom.com/)
     *                      关于期货柜台的错误码说明
     * @param cancelOrder   撤单回报对象，同onExecOrder接口说明
     */
    virtual void onCancelExecOrder(int errorCode, const XTFOrder *cancelOrder) {}

    /**
     * 询价应答接口
     *
     * 询价请求成功或者失败都会产生该事件，即调用 demandQuote() 方法后，询价结果由此接口通知用户。
     *
     * @param errorCode     操作错误码
     *                      0-表示成功；非0-表示发生错误的错误码；
     *                      具体原因请查阅API说明文档或者官方文档平台(http://docs.accelecom.com/)
     *                      关于期货柜台的错误码说明
     * @param inputQuoteDemand 询价时用户传入的数据
     *                         该对象API本地没有持久化，不能在本接口之外使用此对象。
     *                         可以保存对象副本，做进一步处理。
     */
    virtual void onDemandQuote(int errorCode, const XTFInputQuoteDemand &inputQuoteDemand) {}

    /**
     * 持仓组合回报事件通知接口
     *
     * 如果启用组合功能，当持仓被组合或解锁组合时，会产生组合回报事件，通过此接口通知用户。
     * API本地没有持久化组合回报事件对象，用户收到事件后，应立即处理对应的持仓组合逻辑。
     *
     * @param errorCode     操作错误码
     *                      0-表示成功；非0-表示发生错误的错误码；
     *                      具体原因请查阅API说明文档或者官方文档平台(http://docs.accelecom.com/)
     *                      关于期货柜台的错误码说明
     * @param combEvent     持仓组合回报事件对象
     *                      该对象API本地没有持久化，不能在本接口之外使用对象指针。
     *                      可以保存对象副本，做进一步处理。
     */
    virtual void onPositionCombEvent(int errorCode, const XTFPositionCombEvent &combEvent) {}


    //////////////////////////////// 数据变化通知接口 ////////////////////////////////

    /**
     * 账户发生变化时回调该接口
     *
     * event: 账户变化事件类型
     * - XTF_EVT_AccountCashInOut 账户资金发生出入金流水变化
     *
     * action: 账户变化的动作
     * - XTF_CASH_In 入金
     * - XTF_Cash_Out 出金
     *
     */
    virtual void onAccount(int event, int action, const XTFAccount *account) {}

    /**
     * 交易所前置状态发生变化时回调该接口
     *
     * event:
     * - XTF_EVT_ExchangeChannelConnected
     * - XTF_EVT_ExchangeChannelDisconnected
     */
    virtual void onExchange(int event, int channelID, const XTFExchange *exchange) {}

    /**
     * 合约发生变化时回调该接口
     *
     * event:
     * - XTF_EVT_InstrumentStatusChanged
     */
    virtual void onInstrument(int event, const XTFInstrument *instrument) {}



    //////////////////////////////// 外接行情接口 ////////////////////////////////

    /**
     * 行情发生变化时回调该接口
     *
     * 默认只通知用户subscribe()指定的合约行情，参见subscribe()接口。
     */
    virtual void onBookUpdate(const XTFMarketData *marketData) {}



    //////////////////////////////// 其他通用接口 ////////////////////////////////

    /**
     * 事件通知接口
     *
     * 事件类型：
     *
     * @param event 事件对象
     *              该对象API本地没有持久化，不能在本接口之外使用对象指针。
     *              可以保存对象副本，做进一步处理。
     */
    virtual void onEvent(const XTFEvent &event) {}

    /**
     * 其他错误通知接口
     *
     * 错误说明：
     * - ERRXTFAPI_RetryMaxCount: TCP重连失败且已达最大次数
     * - ERRXTFAPI_SequenceNotContinous: 数据流水序列号不连续
     * - ERRXTFAPI_OrderCreateFailed: 创建报单对象失败
     * - ERRXTFAPI_ExecOrderCreateFailed: 创建行权/对冲报单对象失败
     * - ERRXTFAPI_InstrumentNotFound: 没有找到关联的合约
     * - ERRXTFAPI_UnzipInProgress: 正在处理数据流解压，不能处理新的数据流
     * - ERRXTFAPI_UnzipBufferEmpty: 解压处理缓冲区为空
     * - ERRXTFAPI_UnzipBufferTooLarge: 解压处理缓冲区过大
     * - ERRXTFAPI_UnzipFailed: 解压失败
     *
     * @param errorCode 错误码 具体原因请查阅API说明文档或者官方文档平台(http://docs.accelecom.com/)
     *                        关于期货柜台的错误码说明
     * @param data 附带的错误数据，无数据则为nullptr
     *             此处的指针不能异步使用，不能在本接口之外使用对象指针。
     *             如果需要异步处理数据，需要创建副本。
     * @param size 错误数据大小，无数据则为0
     */
    virtual void onError(int errorCode, void *data, size_t size) {}
};

class XTF_API_EXPORT XTFApi {
public:
    virtual ~XTFApi() = default;

    ///////////////////////////// 生命期管理接口 ////////////////////////////////

    /**
     * 启动API
     *
     * 调用该接口，API会自动向交易柜台发起连接。
     *
     * 交易柜台的配置信息，默认从创建API对象的配置文件中读取。
     * 也可以通过 setConfig() 接口配置。
     *
     * 如果接口调用成功，将回调 XTFListener::onStart() 接口通知启动的结果。
     *
     * 如果API停止后再重新启动，可以传入新的对象，以处理新的业务逻辑。
     * 例如：
     * XTFSpiA *a = new XTFSpiA()
     * api->start(a);
     * ... // do something.
     * api->stop();
     *
     * XTFSpiB *b = new XTFSpiB()
     * api->start(b); // ok.
     * ... // do something.
     * api->stop();
     *
     * 说明：不建议传入不同的XTFSpi对象，从逻辑处理上存在一定风险。
     *
     * @param spi 回调事件处理对象
     * @return 接口调用成功返回0，否则返回错误码
     */
    virtual int start(XTFSpi *spi) = 0;

    /**
     * 停止API接口
     *
     * 调用该接口，API会断开与交易柜台的连接。
     * 如果接口调用成功，将回调 XTFSpi::onStop() 接口通知停止的结果。
     *
     * 停止后的API接口，可以重新启动。
     * 此时，可以传入新的XTFSpi对象，处理不同的逻辑。
     *
     * @return 接口调用成功返回0，否则返回错误码
     */
    virtual int stop() = 0;



    ///////////////////////////// 会话管理接口 ////////////////////////////////

    /**
     * 登录交易柜台
     *
     * 接口调用成功后，将回调 onLogin() 接口通知登录结果。
     *
     * 登录接口有三种不同形式的重载：
     * 1、不带任何参数的接口，默认使用配置文件中的设置，或者通过 setConfig() 接口设置的信息；
     * 2、仅带有账户和密码的接口，AppID和AuthCode默认使用配置文件的设置。或者通过setConfig() 接口设置的信息；
     * 3、带有全部参数的接口，将会自动覆盖配置文件或 setConfig() 接口设置的信息；
     *
     * @param accountID 资金账户编码
     * @param password  资金账户密码
     * @param appID     应用程序ID
     * @param authCode  认证授权码
     * @return 接口调用成功返回0，否则返回错误码
     */
    virtual int login() = 0;
    virtual int login(const char *accountID, const char *password) = 0;
    virtual int login(const char *accountID, const char *password, const char *appID, const char *authCode) = 0;

    /**
     * 登出交易柜台
     *
     * 该接口调用成功后，将回调 onLogout() 接口通知登录结果。
     *
     * @return 接口调用成功返回0，否则返回错误码
     */
    virtual int logout() = 0;

    /**
     * 修改账户密码
     *
     * 接口调用成功后，将回调 onChangePassword() 接口通知修改密码结果。
     *
     * @return 接口调用成功返回0，否则返回错误码
     *         调用成功并不表示密码修改成功。
     */
    virtual int changePassword(const char *oldPassword, const char *newPassword) = 0;



    //////////////////////////////// 交易接口 ////////////////////////////////

    /**
     * 发送报单
     *
     * 说明：
     * - 该接口默认是线程不安全的，不能在多线程调用同一个API实例对象的报单接口；
     *   如果需要启用线程安全模式，使用setConfig("ORDER_MT_ENABLED", "true")进行配置即可；
     * - 在 XTFSpi::onOrder() 接口中通知用户插入报单的结果和报单状态；
     *
     * @param inputOrder 待插入的报单参数
     * @param inputOrders 待批量插入的报单参数数组
     * @param orderCount 批量插入数量
     * @return 接口调用成功返回0，否则返回错误码
     *         调用成功并不表示报单操作的结果。
     *         ERRXTFAPI_InvalidImp
     *         ERRXTFAPI_NotStarted
     *         ERRXTFAPI_NotLoggedIn
     *         ERRXTFAPI_InvalidInstrument
     *         ERRXTFAPI_OrderCountExceeded
     *         其他，网络发送可能出现的错误码
     */
    virtual int insertOrder(const XTFInputOrder &inputOrder) = 0;
    virtual int insertOrders(const XTFInputOrder inputOrders[], size_t orderCount) = 0;

    /**
     * 发送撤单
     *
     * 说明：
     * - 该接口默认是线程不安全的，不能在多线程调用同一个API实例对象的撤单接口；
     *   如果需要启用线程安全模式，使用setConfig("ORDER_MT_ENABLED", "true")进行配置即可；
     * - 如果使用XTFOrder报单对象直接撤单，务必使用API返回的XTFOrder对象指针，
     *   用户不能从外部构造一个XTFOrder来撤单；
     * - 在 XTFSpi::onCancelOrder() 接口中通知撤单的结果和报单状态；
     * - 不建议使用交易所单号撤单；
     *
     * @param order 待撤销的报单对象
     * @param orderIDType 通过报单编号撤单时，编号的类型，支持：本地单号、柜台流水号、交易所单号
     * @param orderID 通过报单编号撤单时，待撤销的报单编号
     * @param orders 待批量撤销的报单对象数组
     * @param orderCount 批量撤单数量
     * @param exchange 交易所对象
     * @return 接口调用成功返回0，否则返回错误码
     *         调用成功并不表示报单操作的结果。
     *         ERRXTFAPI_InvalidParameter
     *         ERRXTFAPI_InvalidImp
     *         ERRXTFAPI_NotStarted
     *         ERRXTFAPI_NotLoggedIn
     *         ERRXTFAPI_OrderNotFound
     *         ERRXTFAPI_OrderStatusNotAllowedCancel
     *         ERRXTFAPI_OrderCountExceeded
     *         其他，网络发送可能出现的错误码
     */
    virtual int cancelOrder(const XTFOrder *order) = 0;
    virtual int cancelOrder(XTFOrderIDType orderIDType, long orderID) = 0;
    virtual int cancelOrder(XTFOrderIDType orderIDType, long orderID, const XTFExchange *exchange) = 0;
    virtual int cancelOrder(const XTFInputAction &action, const XTFExchange *exchange) = 0;
    virtual int cancelOrders(const XTFOrder *orders[], size_t orderCount) = 0;
    virtual int cancelOrders(XTFOrderIDType orderIDType, long orderIDs[], size_t orderCount) = 0;
    virtual int cancelOrders(const XTFInputAction actions[], size_t actionCount, const XTFExchange *exchange) = 0;



    //////////////////////////////// 行权/对冲接口 ////////////////////////////////

    /**
     * 发送行权/对冲报单
     *
     * 说明：
     * - 行权/对冲的本地报单编号与普通订单的本地报单编号，不能冲突，需要进行统一的编号；
     * - 该接口默认是线程不安全的，不能在多线程调用同一个API实例对象的报单接口；
     *   如果需要启用线程安全模式，使用setConfig("ORDER_MT_ENABLED", "true")进行配置即可；
     * - 在 XTFSpi::onExecOrder() 接口中通知用户插入报单的结果和报单状态；
     *
     * @param execOrder 待插入的报单参数
     * @return 接口调用成功返回0，否则返回错误码
     *         调用成功并不表示报单操作的结果。
     *         ERRXTFAPI_InvalidImp
     *         ERRXTFAPI_NotStarted
     *         ERRXTFAPI_NotLoggedIn
     *         ERRXTFAPI_InvalidInstrument
     *         其他，网络发送可能出现的错误码
     */
    virtual int insertExecOrder(const XTFExecOrder &execOrder) = 0;

    /**
     * 发送行权/自对冲撤单
     *
     * 说明：
     * - 该接口默认是线程不安全的，不能在多线程调用同一个API实例对象的撤单接口；
     *   如果需要启用线程安全模式，使用setConfig("ORDER_MT_ENABLED", "true")进行配置即可；
     * - 如果使用XTFOrder报单对象直接撤单，务必使用API返回的XTFOrder对象指针，
     *   用户不能从外部构造一个XTFOrder来撤单；
     * - 在 XTFSpi::onCancelExecOrder() 接口中通知撤单的结果和报单状态；
     * - 不建议使用交易所单号撤单；
     *
     * @param order 待撤销的报单对象
     * @param orderIDType 通过报单编号撤单时，编号的类型，支持：本地单号、柜台流水号、交易所单号
     * @param orderID 通过报单编号撤单时，待撤销的报单编号
     * @param exchange 交易所对象
     * @return 接口调用成功返回0，否则返回错误码
     *         调用成功并不表示报单操作的结果。
     *         ERRXTFAPI_InvalidParameter
     *         ERRXTFAPI_InvalidImp
     *         ERRXTFAPI_NotStarted
     *         ERRXTFAPI_NotLoggedIn
     *         ERRXTFAPI_OrderNotFound
     *         ERRXTFAPI_OrderStatusNotAllowedCancel
     *         其他，网络发送可能出现的错误码
     */
    virtual int cancelExecOrder(const XTFOrder *order) = 0;
    virtual int cancelExecOrder(XTFOrderIDType orderIDType, long orderID) = 0;
    virtual int cancelExecOrder(XTFOrderIDType orderIDType, long orderID, const XTFExchange *exchange) = 0;


    /**
     * 发送询价请求
     *
     * 说明：
     * - 该接口默认是线程不安全的，不能在多线程调用同一个API实例对象的询价接口；
     *   如果需要启用线程安全模式，使用setConfig("ORDER_MT_ENABLED", "true")进行配置即可；
     * - 在 XTFSpi::onDemandQuote() 接口中通知询价的结果。
     *
     * @param instrument 待询价的合约对象
     * @param localID 询价请求的本地编号，无规则约束，由用户自定义。可选参数，默认为0
     * @param inputQuoteDemand 询价请求对象
     * @return 接口调用成功返回0，否则返回错误码
     *         调用成功并不表示询价操作的结果。
     *         ERRXTFAPI_InvalidParameter
     *         ERRXTFAPI_InvalidImp
     *         ERRXTFAPI_NotStarted
     *         ERRXTFAPI_NotLoggedIn
     *         其他，网络发送可能出现的错误码
     */
    virtual int demandQuote(const XTFInstrument *instrument, int localID = 0) = 0;
    virtual int demandQuote(const XTFInputQuoteDemand &inputQuoteDemand) = 0;


    /**
     * 发送持仓组合/拆分请求
     *
     * 说明：
     * - 在 XTFSpi::onPositionCombEvent() 接口中通知持仓组合/拆分的结果。
     *
     * @param inputCombPosition 持仓组合请求对象
     * @return 接口调用成功返回0，否则返回错误码
     *         调用成功并不表示询价操作的结果。
     *         ERRXTFAPI_InvalidParameter
     *         ERRXTFAPI_InvalidImp
     *         ERRXTFAPI_NotStarted
     *         ERRXTFAPI_NotLoggedIn
     *         其他，网络发送可能出现的错误码
     */
    virtual int combinePositions(const XTFInputCombPosition &inputCombPosition) = 0;



    //////////////////////////////// 外接行情接口 ////////////////////////////////

    /**
     * 订阅行情
     *
     * 行情订阅成功后，当收到对应合约的行情数据后，会回调 onBookUpdate() 接口。
     *
     * @param instrument    合约对象，空指针表示订阅所有合约对象
     * @return              接口调用成功或失败，调用成功表示行情订阅成功。
     */
    virtual int subscribe(const XTFInstrument *instrument) = 0;

    /**
     * 取消订阅行情
     *
     * @param instrument    合约对象，空指针表示取消订阅所有合约对象
     * @return              接口调用成功或失败，调用成功表示行情订阅取消成功。
     */
    virtual int unsubscribe(const XTFInstrument *instrument) = 0;

    /**
     * 更新合约行情
     *
     * 暂时仅支持一档行情数据。
     *
     * 更新行情后，API会自动处理以下逻辑：
     * 1、根据合约仓位计算持仓盈亏；
     * 2、如果用户调用subscribe()接口订阅行情数据，会通过 XTFSpi::onBookUpdate() 接口通知用户；
     *
     * @return 接口调用成功或失败，调用成功表示行情更新成功。
     */
    virtual int updateBook(const XTFInstrument *instrument,     ///< 合约对象
                           double lastPrice,                    ///< 最新价
                           double bidPrice,                     ///< 买入价。为零代表无买入价。
                           int    bidVolume,                    ///< 买入量。为零代表无买入量。
                           double askPrice,                     ///< 卖出价。为零代表无卖出价。
                           int    askVolume                     ///< 卖出量。为零代表无卖出量。
    ) = 0;



    //////////////////////////////// 辅助功能接口 ////////////////////////////////

    /**
     * 同步柜台资金
     *
     * 本接口使用同步方式向柜台查询资金。
     *
     * 调用约束：
     * - 本接口不能并发调用，上一次同步请求结束后，才能进行下一次调用；
     * - 调用接口会阻塞调用者线程，直到应答返回或超时返回；
     *
     * 使用建议：
     * - 如果没有外部行情接入，调用此接口可以同步柜台和本地计算的资金差异；
     * - 如果有外部极速行情，建议使用 updateBook() 的方式，在本地计算资金；
     * - 接入外部行情和调用资金同步接口，两种方式不要混用，以免出现资金错误；
     *
     * @param msTimeout 超时时间，单位：毫秒；默认为45毫秒超时，<=0 表示不允许超时
     * @return          0：表示资金同步成功，可以访问XTFAccount对象查看最新的资金数据；
     *                  ETIMEDOUT：表示请求超时；
     *                  <0：表示内部发生错误，可以联系技术支持查看具体错误信息；
     */
    virtual int syncFunds(int msTimeout = 45) = 0;

    /**
     * 传入合约参数，创建一个预热报单，写入用户提供的缓冲区之中。
     *
     * 约束说明：
     * - 发送至柜台的预热报单每秒限制不超过50个，建议发送间隔为：20ms~50ms；
     * - 预热报单和真实报单建议使用同一个线程发送；
     *
     * 默认创建的预热报单没有合约信息，如果需要携带合约信息，传入合约参数即可。
     * 合约可以通过getInstrumentByID()接口查询获得。
     *
     * @param buf           用户提供的预热报单缓冲区
     * @param size          用户提供的预热报单缓冲区大小，不小于64字节。如果大于64字节，那么仅头部的64字节有效。
     * @param instrument    合约指针，默认为空
     * @return              0-表示成功，非0表示对应的错误码
     */
    virtual int buildWarmOrder(void *buf, size_t size, const XTFInstrument *instrument = nullptr) = 0;



    //////////////////////////////// 查询管理接口 ////////////////////////////////

    /**
     * 获取当前资金账户信息
     * @return 返回资金账户对象指针
     */
    virtual const XTFAccount*       getAccount() = 0;

    /**
     * 获取交易所信息
     */
    virtual int                     getExchangeCount() = 0;
    virtual const XTFExchange*      getExchange(int pos) = 0;

    /**
     * 获取合约信息
     */
    virtual int                     getInstrumentCount() = 0;
    virtual const XTFInstrument*    getInstrument(int pos) = 0;
    virtual const XTFInstrument*    getInstrumentByID(const char *instrumentID) = 0;

    /**
     * 获取组合合约信息
     */
    virtual int                     getCombInstrumentCount() = 0;
    virtual const XTFCombInstrument*getCombInstrument(int pos) = 0;
    virtual const XTFCombInstrument*getCombInstrumentByID(const char *combInstrumentID) = 0;

    /**
     * 获取品种信息
     */
    virtual int                     getProductCount() = 0;
    virtual const XTFProduct*       getProduct(int pos) = 0;
    virtual const XTFProduct*       getProductByID(const char *productID) = 0;

    /**
     * 根据指定的过滤条件查询报单
     *
     * 查询所有满足条件的报单可以使用下面的方法：
     * XTFOrderFilter filter;
     * filter.orderStatus[0] = XTF_OS_Queuing;     // 查询正在排队的报单
     * filter.orderStatus[1] = XTF_OS_PartTraded;  // 查询已部分成交的报单
     * ... // set filter conditions.
     *
     * int count = api->findOrders(filter, 0, nullptr);
     * if (count < 0) {
     *     ... // error.
     *     return;
     * }
     *
     * const XTFOrder **orders = new const XTFOrder *[count];
     * int result = api->findOrders(filter, count, orders);
     * if (result < 0) {
     *     ... // error.
     * } else {
     *     ... // find ok.
     * }
     *
     * @param filter 过滤条件
     * @param count  指定最大的查询结果数量
     * @param orders 保存查询结果的缓冲区(数组)
     * @return 实际的查询结果数量，负数表示查询发生错误
     */
    virtual int findOrders(const XTFOrderFilter &filter, unsigned int count, const XTFOrder *orders[]) = 0;

    /**
     * 根据指定的过滤条件查询成交
     *
     * 查询所有满足条件的成交可以使用下面的方法：
     * XTFTradeFilter filter;
     * ... // set filter.
     *
     * int count = api->findTrades(filter, 0, nullptr);
     * const XTFTrade **trades = new const XTFTrade *[count];
     * int result = api->findTrades(filter, count, trades);
     * if (result < 0) {
     *     ... // error.
     * } else {
     *     ... // find ok.
     * }
     *
     * @param filter 过滤条件
     * @param count  指定最大的查询结果数量
     * @param trades 保存查询结果的缓冲区(数组)
     * @return 实际的查询结果数量，负数表示查询发生错误
     */
    virtual int findTrades(const XTFTradeFilter &filter, unsigned int count, const XTFTrade *trades[]) = 0;



    //////////////////////////////// 参数配置管理接口 ////////////////////////////////

    /**
     * 启停仓位自动组合功能
     *
     * 创建API实例后、启动API前，调用 enableAutoCombinePosition(true) 接口设置启用仓位自动组合功能。
     * 启用组合之后，API登录会发送自动组合标识到柜台。柜台会根据用户的当前持仓，按照组合规则进行自动组合。
     *
     * 应在启动会话start()之前设置自动组合功能，启动之后设置不会生效。
     * 默认为不启用仓位自动组合功能。
     *
     * 仓位自动组合说明：
     * 1、无论是否启用自动组合功能，上个交易日的历史仓位，柜台都会将满足组合规则的仓位全部组合；
     * 2、日内交易的新开仓位，如果用户登录时启用了仓位自动组合，那么剩余未组合的历史仓位（单腿）和日内新开仓位，柜台会将满足组合规则的仓位全部组合；
     * 3、如果日内有多次登录，且最近一次没有启用仓位自动组合，那么上一次登录时，已组合的仓位不再变化，本次会话过程中的新开仓位，柜台不会自动组合。
     *
     * @param enabled true-启用 false-不启用
     */
    virtual void enableAutoCombinePosition(bool enabled) = 0;

    /**
     * 设置会话参数
     *
     * 应在启动会话之前设置参数，启动之后设置的参数不会生效。
     *
     * 会话参数名称列表：
     * "ACCOUNT_ID"：资金账户ID
     * "ACCOUNT_PWD"：资金账户密码
     * "APP_ID"：应用程序ID
     * "AUTH_CODE"：认证授权码
     * "TRADE_SERVER_IP"：交易服务地址
     * "TRADE_SERVER_PORT"：交易服务端口
     * "QUERY_SERVER_IP"：查询服务地址
     * "QUERY_SERVER_PORT"：查询服务端口
     * "HEARTBEAT_INTERVAL"：心跳间隔时间，单位：毫秒
     * "HEARTBEAT_TIMEOUT"：心跳超时时间，单位：毫秒
     * "TCP_RECONNECT_ENABLED"：TCP断开后是否重连
     * "TCP_RETRY_INTERVAL"：TCP重连最小间隔时间，单位：毫秒
     * "TCP_RETRY_INTERVAL_MAX"：TCP重连最大间隔时间，单位：毫秒
     * "TCP_RETRY_COUNT"：TCP重连次数
     * "TCP_WORKER_CORE_ID"：数据收发线程绑核
     * "TCP_WORKER_BUSY_LOOP_ENABLED"：数据收发线程是否启用BusyLoop模式运行
     * "TRADE_WORKER_CORE_ID"：预热报单线程绑核
     * "TASK_WORKER_CORE_ID"：通用任务线程绑核
     * "POSITION_CALC_ENABLED"：是否计算仓位
     * "MONEY_CALC_ENABLED"：是否计算资金，如果启用资金计算，会默认启用仓位计算
     * "HISTORY_ONLY_CALC_ENABLED"：是否仅计算历史流水的资金和仓位，如果启用，那么历史流水追平后，将不再计算资金和仓位
     * "WARM_INTERVAL"：预热时间间隔，取值范围：[10,50]，单位：毫秒
     * "WARM_ENABLED"：预热报单功能是否启用，取值范围：[true,false]，默认为true：表示启用
     * "ORDER_MT_ENABLED"：是否启用线程安全模式，启用线程安全模式，单个API实例可以支持多个线程同时报单；但是会略微增加报单延时。
     * "TRADE_LOG_ENABLED"：是否启用交易明细日志功能。true表示启用，false表示不启用，默认为不启用。
     *     启用后会在xtf-api-log日志目录下生成xtf-trade-xxx.log日志文件，记录了上下行交易明细数据，可用于调试。
     *     启用该功能会对下行处理速度有影响。生产环境下，不建议启用。
     * "RELEASE_FINISHED_ORDER_ENABLED"：是否启用完结报单内存资源自动回收功能。true表示启用，false表示不启用，默认为不启用。
     *     启用此功能需要非常谨慎。如果报单量非常大，且无成交的撤单占据大部分，那么可以启用此功能。其他场景下，建议不要启用。
     *     启用后对于所有的错单、拒单和撤单（无成交），其分配的内存资源会被自动回收。
     *     如果启用此功能，报单内存可能会被回收利用，用户需要根据报单回报消息和报单状态变化，来判断XTFOrder对象指针是否有效，
     *     只有状态是已接收、正在队列中、部分成交和全部成交的报单，内存资源会持久保留、不会回收，这些报单的指针是有效的。
     *     其他状态的报单指针是无效的，需要丢弃。
     * "HISTORICAL_FINISHED_ORDER_FILTER_ENABLED"：是否启用历史完结订单过滤功能（仅做市商柜台支持此功能）。true表示启用，false表示不启用，默认为不启用。
     *     如果启用此过滤功能，客户端登录时，对所有的历史回报流水做以下过滤处理：
     *     1、普通单：错单和撤单（无成交）的回报流水，不再推送给客户端；
     *     2、报价单：状态为错单、两腿衍生单都是撤单且无成交的回报流水，不再推送给客户端；
     *     3、衍生单：所有报价产生的衍生单，错单和撤单（无成交）的回报流水，不再推送给客户端；
     *     如果启用此过滤功能，客户端可以收到的历史回报流水数据包括：
     *     1、普通单：状态是已接收、正在队列中、部分成交、全部成交和部分成交的撤单的回报流水；
     *     2、衍生单及其原生报价单：状态是已接收、正在队列中、部分成交、全部成交和部分成交的撤单的回报流水及其相关的报价单流水；
     *     3、其他不在上述过滤条件中的回报流水；
     *     此功能不影响登录后的回报流水，登录后的所有报撤单流水，全量返回给客户端。
     * "RUN_MODE"：运行模式参数，0-表示正常模式，1-表示极简模式。
     *     极简模式可以加速下行回报的处理性能，但部分API接口因为数据处理逻辑的缺失，而无法使用。
     *     对于那些非常关注下行处理耗时的应用场景，可以启用此模式。此模式下的功能变化包括：
     *     1、用户报单后，报单回报不再分配内存，直接将回报字段转换成XTFOrder对象后，通知用户处理，XTFOrder对象回调结束后失效；
     *     2、成交回报与报单回报类似，不再分配内存，直接将成交回报字段转换成XTFTrade对象后，通知用户处理，XTFTrade对象回调结束后失效；
     *     3、可能存在乱序问题，需要用户自行处理。乱序场景包括：报单应答和报单回报乱序、报单回报和成交回报乱序；
     *     4、原有的API撤单接口失效，只能通过编号（系统流水号或本地单号）+交易所来撤单。本地单号撤单，需要用户具备权限才能撤单；
     *     5、API本地不保存任何的回报和成交数据，因此无法提供报单和成交的数据查询功能，需要用户自己保存所有的报单和成交数据；
     *     6、静态数据和日初资金数据，用户可以直接从API查询获得，持仓、资金、流水、成交回报等信息无法查询；
     * "XNIC_ENABLED"：是否启用低延迟网卡功能。
     *     如果启用此配置项，那么API启动时会自动检测当前是否有可用的低延迟网卡，如果有则使用低延迟网卡快速通道进行报撤单。
     *     如果禁用此配置项，那么API启动时不会检测是否有可用的低延迟网卡，默认走系统协议栈通道进行报撤单。
     *     若启用该功能，建议使用onload-7.1.x版本或exanic2.7.x版本的驱动。
     *     默认为启用。
     * "XNIC_NAME"：指定启用的低延迟网卡名称。
     *     如果启用此配置项，则使用指定的网卡；反之，则自动选择一个可用的网卡。
     *     建议使用默认配置，由API自动检测可用网卡。
     *     @since 4.1.1350
     * "XNIC_TYPE"：指定启用的低延迟网卡类型。
     *     如果启用此配置项，则使用指定类型的驱动；反之，则自动选择匹配的网卡驱动。
     *     支持类型范围：[sfc, exanic]，建议使用默认配置，由API自动检测可用网卡类型。
     *     @since 4.1.1350
     * "ORDER_GROUP_ID"：配置用户报单分组编号，由用户自定义。取值范围：[0, 128]，默认为0。
     * "COMBINATION_ENABLED"：是否启用仓位的自动组合功能。
     *     true表示启用，false表示不启用，默认为不启用。
     *     @since 4.1.1225
     * "COMBINATION_TYPES"：指定自动组合的类型列表。默认表示自动组合所有的类型。
     *     组合类型取值范围请参考XTFDataTypes.h文件中XTFCombType的枚举定义。
     *     组合类型列表的每个元素，以逗号','分割。
     *     例如：针对大商和广期所，可以按照如下配置自动组合的类型：
     *     - 自动组合期货对锁、期权对锁：COMBINATION_TYPES=0,1
     *     - 自动组合期货对锁、跨期套利和跨品种套利：COMBINATION_TYPES=0,2,3
     *     注意：此配置选项需要与自动组合开关 COMBINATION_ENABLED 选项配合使用
     *     @since 4.1.1225
     * "CHECK_VERSION_ENABLED"：是否启用头文件和库的版本一致性检测。
     *     如果启用此配置项，那么API启动时会自动检测当前使用的头文件版本和库版本是否一致。如果不一致则启动失败，并返回相应的错误码。
     *     如果禁用此配置项，那么API启动时不会检测头文件版本和库版本是否一致。
     *     默认为启用。如需要禁用，请设置此选项为 CHECK_VERSION_ENABLED=false。
     *     @since 4.1.1350
     *
     * "MD_ENABLED"
     *     指定是否启用柜台的TCP行情服务。
     *     true 表示启用行情服务，false 表示不启用。
     *     当启用行情服务时，API登录成功且静态数据推送完成之后，会自动发起行情服务连接。
     *     连接成功后，开始接收柜台定时推送的行情数据。默认情况下，不会回调通知行情数据。
     *     用户调用了 subscribe 接口订阅成功后，API会通过 onBookUpdated 接口通知行情数据。
     *     如果连接失败，API会记录错误信息到日志文件，并无主动通知的接口。
     *     如果用户关注行情且长时间没有收到行情数据，可以检查API日志，以确认行情的错误原因。
     *     默认不启用行情。
     *     @since 4.1.1360
     *
     * "MD_SERVER_IP"
     *     指定柜台的TCP行情服务IP地址
     *     仅当 MD_ENABLED=true 时，配置生效。
     *     @since 4.1.1360
     *
     * "MD_SERVER_PORT"
     *     指定柜台的TCP行情服务端口
     *     仅当 MD_ENABLED=true 时，配置生效。
     *     @since 4.1.1360
     *
     * 用户也可以设置自己的参数，以便在需要的地方使用 getConfig() 接口进行获取。
     * 会话内部不使用这部分用户参数，仅对其临时存储。
     *
     * @param name 参数名称
     * @param value 参数值
     * @return 参数配置成功或失败
     */
    virtual int setConfig(const char *name, const char *value) = 0;

    /**
     * 查询会话参数
     *
     * 参数名称列表： 同上
     *
     * @param name  参数名称，参见setConfig()；
     * @return      参数值
     *              返回的字符串为临时字符串对象，如有需要请保存字符串值，再做后续的处理。
     */
    virtual const char* getConfig(const char *name) = 0;

    /**
     * 设置回报过滤规则（白名单）
     *
     * 约束：
     * - 必须在调用登录接口 login() 之前，设置回报过滤规则才能生效；
     *
     * @param filter 回报过滤规则
     * @return 0表示设置成功，非0表示本次操作失败对应的错误码
     */
    virtual int setReportFilter(const XTFReportFilter &filter) = 0;

    /**
     * 查询生效的回报过滤规则（白名单）
     *
     * 约束：
     * - 必须在会话登录期间的调用，才是有效的；
     * - 会话退出时，会自动清除所有规则；
     *
     * @param filter 回报过滤规则
     * @return 0表示查询成功，非0表示本次查询失败对应的错误码
     */
    virtual int getReportFilter(XTFReportFilter &filter) = 0;

    /**
     * 投资者指定席位优先级
     *
     * 约束：
     * - 需要配置权限后才能指定席位优先级；
     *
     * @param exchange：待设置的交易所对象。如果柜台是单交易所，该参数可以传入null表示默认的交易所对象；
     * @param priorities：席位优先级数组，传递每个席位的优先级权重值；
     * @param prioritiesCount：席位优先级数组长度，用来和交易所席位数进行比较。
     *                         1）如果数组长度大于实际的席位数，则只取前面的席位数量对应的优先级；
     *                         2）如果数组长度小于实际的席位数，则后面默认的席位优先级为0；
     *                         3）prioritiesCount的最大值为16；
     * @param strategy：1表示优先发送策略，2表示概率发送策略。默认为1；
     * @return 0表示设置成功，非0表示失败对应的错误码
     */
    virtual int setChannelPriorities(const XTFExchange *exchange, uint32_t *priorities, size_t prioritiesCount, XTFChannelSelectionStrategy strategy) = 0;

    /**
     * 获取API版本
     *
     * @return 版本字符串
     */
    virtual const char *getVersion() = 0;
};

#ifdef __cplusplus
extern "C" {
#endif

/**
 * 创建API实例对象
 *
 * 有两种方式创建API实例：
 * 1. 配置文件方式，需要传入配置文件路径。如果传入的路径打开失败，那么创建API实例失败，返回空指针；
 * 2. 参数设置方式，不需要传入配置文件路径。调用函数时配置文件路径参数传入空指针（nullptr）即可。
 *    创建成功后，需要通过API->setConfig()接口设置以下必选参数：
 *    "ACCOUNT_ID"：资金账户ID
 *    "ACCOUNT_PWD"：资金账户密码
 *    "APP_ID"：应用程序ID
 *    "AUTH_CODE"：认证授权码
 *    "TRADE_SERVER_IP"：交易服务地址
 *    "TRADE_SERVER_PORT"：交易服务端口
 *    "QUERY_SERVER_IP"：查询服务地址
 *    "QUERY_SERVER_PORT"：查询服务端口
 *    详细内容请参考API->setConfig()接口注释说明。
 *
 * @param configPath 参数配置文件路径
 * @return API实例对象指针
 */
XTF_API_EXPORT XTFApi*      makeXTFApi(const char *configPath);

/**
 * 查询API版本号
 *
 * @return API版本号字符串
 */
XTF_API_EXPORT const char*  getXTFVersion();

/**
 * 根据错误码查询错误消息描述
 *
 * @param errorCode 错误码
 * @param language  语言类型；0-中文，1-英文
 * @return 错误消息描述
 */
XTF_API_EXPORT const char*  getXTFErrorMessage(int errorCode, int language);

/**
 * 按纳秒获取当前时间戳
 *
 * @return 纳秒计数的时间戳
 */
XTF_API_EXPORT unsigned long long getXTFNanoTimestamp();

/**
 * 是否启用日志功能
 *
 * @param enabled
 * - true: 启用日志
 * - false: 禁用日志
 */
XTF_API_EXPORT void setXTFLogEnabled(bool enabled);

#ifdef __cplusplus
}
#endif

#endif
