﻿/*! \file EesTraderApi.h
 *  \brief EES交易客户端头文件   
 *  
 *  该文档详细描述了EES交易客户端的接口定义。
*/
#pragma  once 
#include "EesTraderDefine.h"
#include "EesTraderErr.h"

#ifdef WIN32
	#ifdef SL_EES_TRADE_EXPORTS
		#define SL_EES_TRADE_CLASS __declspec(dllexport)	  
		#define SL_EES_TRADE_FUN		extern "C" __declspec(dllexport)
	#else
		#define SL_EES_TRADE_CLASS __declspec(dllimport)	  
		#define SL_EES_TRADE_FUN		extern "C" __declspec(dllimport)
	#endif

	/// \brief EES交易客户端动态库名
	#define EES_TRADER_DLL_NAME    "EESTraderApi.dll"
	/// \brief EES交易客户端静态库名
	#define EES_TRADER_LIB_NAME    "EESTraderApi.lib"

	#include <windows.h>

#else	
	#define SL_EES_TRADE_CLASS 
	#define SL_EES_TRADE_FUN  extern "C" 

	#ifndef OUT
		#define OUT
	#endif

	#ifndef NULL
		#define NULL 0
	#endif

	/// \brief EES交易客户端动态库名
	#define EES_TRADER_DLL_NAME    "libEESTraderApi.so"

#endif

/// \brief EES交易客户端回调基类
class  EESTraderEvent
{
public:	
	virtual ~EESTraderEvent()
	{

	}
	/// 连接消息的回调
	/**
		\brief	服务器主动断开，也会收到这个消息
		\param  errNo                   连接成功能与否的消息
		\param  pErrStr                 错误信息
		\return void  
	*/
	virtual void OnConnection(ERR_NO errNo, const char* pErrStr ){}

	/// 连接断开消息的回调
	/**
		\brief	服务器主动断开，会收到这个消息
		\param  ERR_NO errNo         连接成功能与否的消息
		\param  const char* pErrStr  错误信息
		\return void  
	*/
	virtual void OnDisConnection(ERR_NO errNo, const char* pErrStr ){}

	/// 登录消息的回调
	/**
		\param  pLogon                  登录成功或是失败的结构
		\return void 
	*/
	virtual void OnUserLogon(EES_LogonResponse* pLogon){}

	/// 查询用户下面帐户的返回事件
	/**
		\param  pAccountInfo	        帐户的信息
		\param  bFinish	                如果没有传输完成，这个值是 false ，如果完成了，那个这个值为 true 
		\remark 如果碰到 bFinish == true，那么是传输结束，并且 pAccountInfo值无效。
		\return void 
	*/
	virtual void OnQueryUserAccount(EES_AccountInfo * pAccoutnInfo, bool bFinish){}

	/// 查询帐户下面仓位信息的返回事件
	/**
		\param  pAccount	                帐户ID 	
		\param  pAccoutnPosition	        帐户的仓位信息					   
		\param  nReqId		                发送请求消息时候的ID号。
		\param  bFinish	                    如果没有传输完成，这个值是false，如果完成了，那个这个值为 true 
		\remark 如果碰到 bFinish == true，那么是传输结束，并且 pAccountInfo值无效。
		\return void 
	*/
	virtual void OnQueryAccountPosition(const char* pAccount, EES_AccountPosition* pAccoutnPosition, int nReqId, bool bFinish){}	


	/// 查询帐户下面资金信息的返回事件
	/**
		\param  pAccount	                帐户ID 	
		\param  pAccoutnPosition	        帐户的仓位信息					   
		\param  nReqId		                发送请求消息时候的ID号
		\return void 
	*/
	virtual void OnQueryAccountBP(const char* pAccount, EES_AccountBP* pAccoutnPosition, int nReqId ){}	

	/// 查询合约列表的返回事件
	/**
		\param  pSymbol	                    合约信息   
		\param  bFinish	                    如果没有传输完成，这个值是 false，如果完成了，那个这个值为 true   
		\remark 如果碰到 bFinish == true，那么是传输结束，并且 pSymbol 值无效。
		\return void 
	*/
	virtual void OnQuerySymbol(EES_SymbolField* pSymbol, bool bFinish){}

	/// 查询帐户交易保证金的返回事件
	/**
	    \param  pAccount                    帐户ID 
		\param  pSymbolMargin               帐户的保证金信息 
		\param  bFinish	                    如果没有传输完成，这个值是 false，如果完成，那个这个值为 true 
		\remark 如果碰到 bFinish == true，那么是传输结束，并且 pSymbolMargin 值无效。
		\return void 
	*/
	virtual void OnQueryAccountTradeMargin(const char* pAccount, EES_AccountMargin* pSymbolMargin, bool bFinish ){}

	/// 查询帐户交易费用的返回事件
	/**
		\param  pAccount                    帐户ID 
		\param  pSymbolFee	                帐户的费率信息	 
		\param  bFinish	                    如果没有传输完成，这个值是 false，如果完成了，那个这个值为 true    
		\remark 如果碰到 bFinish == true ，那么是传输结束，并且 pSymbolFee 值无效。
		\return void 
	*/
	virtual void OnQueryAccountTradeFee(const char* pAccount, EES_AccountFee* pSymbolFee, bool bFinish ){}

	/// 下单被市场接受的事件
	/**
		\brief 表示这个订单已经被交易所正式的接受
		\param  pAccept	                    订单被接受以后的消息体，里面包含了市场订单ID
		\return void 
	*/
	virtual void OnOrderAccept(EES_OrderAcceptField* pAccept ){}

	///	下单被市场拒绝的事件
	/**
		\brief	订单被市场或是柜台系统拒绝，可以查看语法检查或是风控检查。 
		\param  pReject	                    订单被接受以后的消息体，里面包含了市场订单ID
		\return void 
	*/
	virtual void OnOrderReject(EES_OrderRejectField* pReject ){}

	///	订单成交的消息事件
	/**
		\brief	成交里面包括了订单市场ID，建议用这个ID查询对应的订单
		\param  pExec	                   订单被接受以后的消息体，里面包含了市场订单ID
		\return void 
	*/
	virtual void OnOrderExecution(EES_OrderExecutionField* pExec ){}

	///	订单成功撤销事件
	/**
		\brief	成交里面包括了订单市场ID，建议用这个ID查询对应的订单
		\param  pCxled		               订单被接受以后的消息体，里面包含了市场订单ID
		\return void 
	*/
	virtual void OnOrderCxled(EES_OrderCxled* pCxled ){}

	///	撤单被拒绝的消息事件
	/**
		\brief	一般会在发送撤单以后，收到这个消息，表示撤单被拒绝
		\param  pReject	                   撤单被拒绝消息体
		\return void 
	*/
	virtual void OnCxlOrderReject(EES_CxlOrderRej* pReject ){}

	///	查询订单的返回事件
	/**
		\brief	查询订单信息时候的回调，这里面也可能包含不是当前用户下的订单
		\param  pAccount                 帐户ID 
		\param  pQueryOrder	             查询订单的结构
		\param  bFinish	                 如果没有传输完成，这个值是 false，如果完成了，那个这个值为 true    
		\remark 如果碰到 bFinish == true，那么是传输结束，并且 pQueryOrder值无效。
		\return void 
	*/
	virtual void OnQueryTradeOrder(const char* pAccount, EES_QueryAccountOrder* pQueryOrder, bool bFinish  ){} 

	///	查询订单的返回事件
	/**
		\brief	查询订单信息时候的回调，这里面也可能包含不是当前用户下的订单成交
		\param  pAccount                        帐户ID 
		\param  pQueryOrderExec	                查询订单成交的结构
		\param  bFinish	                        如果没有传输完成，这个值是false，如果完成了，那个这个值为 true    
		\remark 如果碰到 bFinish == true，那么是传输结束，并且pQueryOrderExec值无效。
		\return void 
	*/
	virtual void OnQueryTradeOrderExec(const char* pAccount, EES_QueryOrderExecution* pQueryOrderExec, bool bFinish  ){}

	///	接收外部订单的消息
	/**
		\brief	一般会在系统订单出错，进行人工调整的时候用到。
		\param  pPostOrder	                    查询订单成交的结构
		\return void 
	*/
	virtual void OnPostOrder(EES_PostOrder* pPostOrder ){}	

	///	接收外部订单成交的消息
	/**
		\brief	一般会在系统订单出错，进行人工调整的时候用到。
		\param  pPostOrderExecution	             查询订单成交的结构
		\return void 
	*/
	virtual void OnPostOrderExecution(EES_PostOrderExecution* pPostOrderExecution ){}

};

/// \brief EES交易客户端纯虚基类
class SL_EES_TRADE_CLASS EESTraderApi
{
public:
	virtual ~EESTraderApi()
	{
	}

	/// 连接服务器
	/**	
		\param  const char* svrAddr		    服务器的地址
		\param  int nPort					服务端口
		\return RESULT						连接成功或是失败，0表示连接成功，非0表示连接失败。参考 EesTraderErr.h文件
	*/
	virtual RESULT	ConnServer(const char* svrAddr, int nPort, EESTraderEvent* pEvent)  = 0 ;
	
	/// 断开服务器
	/**	
		\return RESULT						参考 EesTraderErr.h文件
	*/
	virtual RESULT	DisConnServer()  = 0 ;

	/// 用户登录
	/**	
		\brief	这里输入的不是帐户名，是用户名
		\param  const char* userId			用户名
		\param  const char* userPwd		    用户登录密码
		\return RESULT			返回值，参考 EesTraderErr.h文件
	*/
	virtual RESULT UserLogon(const char* userId, const char* userPwd )  = 0 ;

	/// 查询合约列表
	/**
		\brief	只会返回当天有效的合约
		\return RESULT			返回值，参考 EesTraderErr.h文件
	*/
	virtual RESULT QuerySymbolList( )  = 0 ;


	/// 查询用户包含的帐户
	/**
		\return RESULT			返回值，参考 EesTraderErr.h文件
	*/
	virtual RESULT QueryUserAccount()  = 0 ;

	/// 查询帐户仓位
	/**
		\brief	
		\param  const char* accountId		帐户ID
		\param  int   nReqId			    请求查询的ID号
		\return RESULT			返回值，参考EesTraderErr.h文件
	*/
	virtual RESULT QueryAccountPosition(const char* accountId, int nReqId)  = 0 ;

	/// 查询帐户BP
	/**
		\brief	
		\param  const char* accountId		帐户ID
		\param  int   nReqId				请求查询的ID号
		\return RESULT			返回值，参考EesTraderErr.h文件
	*/
	virtual RESULT QueryAccountBP(const char* accountId, int nReqId)  = 0 ;

	/// 查询帐户保证金率
	/**
		\param  const char* accountId		帐户ID
		\return RESULT			返回值，参考EesTraderErr.h文件
	*/
	virtual RESULT QueryAccountTradeMargin(const char* accountId )  = 0 ;

	/// 查询帐户手续费
	/**
		\brief	由于，每一个帐户的费率不一样，最好每一个帐户都查询一下。
		\param  const char* accountId		帐户ID
		\return RESULT			返回值，参考EesTraderErr.h文件
	*/
	virtual RESULT QueryAccountTradeFee(const char* accountId ) = 0  ;

	/// 下单
	/**
		\param  EES_EnterOrderField* pOrder	组织好的订单结构体
		\param  OUT int* orderToken			返回的 client token 该值在该用户下面唯一
		\return RESULT			返回值，参考EesTraderErr.h文件
	*/
	virtual RESULT	EnterOrder(EES_EnterOrderField* pOrder, OUT int* orderToken ) = 0  ;

	/// 撤销订单指令
	/**
		\param  EES_CancelOrder* pCxlOrder		撤单结构体
		\return RESULT			返回值，参考EesTraderErr.h文件
	*/
	virtual RESULT	CancelOrder(EES_CancelOrder* pCxlOrder)  = 0 ;

	/// 查询帐户订单信息 
	/**
		\param  const char* accountId		     帐户ID
		\return RESULT			返回值，参考EesTraderErr.h文件
	*/
	virtual RESULT	QueryAccountOrder(const char* accountId )  = 0 ;

	/// 查询帐户成交信息
	/**
		\param  const char* accountId		     帐户ID
		\return RESULT			返回值，参考EesTraderErr.h文件
	*/
	virtual RESULT	QueryAccountOrderExecution(const char* accountId ) = 0 ; 

};

/// 创建EES交易客户端实例的函数名
#define CREATE_EES_TRADER_API_NAME ("CreateEESTraderApi")

/// 销毁EES交易客户端实例的函数名
#define DESTROY_EES_TRADER_API_NAME ("DestroyEESTraderApi")

/// 创建EES交易客户端实例函数声明
SL_EES_TRADE_FUN EESTraderApi* CreateEESTraderApi(void) ; 

/// 销毁EES交易客户端实例函数声明
SL_EES_TRADE_FUN void DestroyEESTraderApi(EESTraderApi* pEESTraderApi) ; 

/// 创建EES交易客户端实例函数指针类型
typedef EESTraderApi* (*funcCreateEESTraderApi)(void) ;

/// 销毁EES交易客户端实例函数指针类型
typedef void (*funcDestroyEESTraderApi)(EESTraderApi* pEESTraderApi) ;

